package com.atlassian.crowd.embedded.validator.rule;

import com.atlassian.crowd.embedded.validator.FieldValidationError;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * Builder class which facilitates creation of {@link ValidationRule}
 *
 * @param <V>
 *         Entity type to be validated
 * @since 3.2.0
 */
public class RuleBuilder<V> {

    /**
     * Implementation class for mapping predicates and value retriever functions
     */
    private final class Condition {
        final Predicate<V> validationFunction;

        <M> Condition(Predicate<M> predicate, Function<V, M> valueFunction) {
            validationFunction = valueFunction.andThen(predicate::test)::apply;
        }

        boolean evaluate(V argument) {
            return validationFunction.test(argument);
        }
    }

    private String fieldName;

    private Function<V, String> message;

    private final List<Condition> conditions = new ArrayList<>();

    /**
     * Utility method which returns the predicate for matching any string value to the given regex pattern.
     *
     * @param pattern
     *         The pattern to which the string is to be tested against
     * @return The generated predicate which tests an input string to specified regex
     */
    public static Predicate<String> regex(String pattern) {
        return value -> value != null && value.matches(pattern);
    }

    public static Predicate<String> notNull() {
        return Objects::nonNull;
    }

    public static Predicate<String> isNull() {
        return Objects::isNull;
    }

    public static Predicate<String> eq(String testValue) {
        return (value) -> Objects.equals(value, testValue);
    }

    public static <K> Predicate<K> not(Predicate<K> predicate) {
        return predicate.negate();
    }

    public static <K> Predicate<K> matchesAny(Predicate<K>... predicates) {
        return Arrays.stream(predicates).reduce(Predicate::or).get();
    }

    public static <K> Predicate<K> matchesAll(Predicate<K>... predicates) {
        return Arrays.stream(predicates).reduce(Predicate::and).get();
    }

    public static Supplier<String> message(I18nHelper i18nHelper, String messageKey) {
        return () -> i18nHelper.getText(messageKey);
    }

    public static Predicate<String> isValidURI() {
        return (value) -> {
            try {
                if (value == null)
                    return false;
                final URI uri = new URI(value);
                return uri.getHost() != null;
            } catch (URISyntaxException e) {
                return false;
            }
        };
    }

    public static Predicate<String> inLongRange(Long min, Long max) {
        return value -> {
            try {
                Long actualValue = Long.parseLong(value);
                return actualValue >= min && actualValue <= max;
            } catch (NumberFormatException e) {
                return false;
            }

        };
    }

    public static Predicate<String> greaterThanOrEquals(Long number) {
        return value -> {
            try {
                Long actualValue = Long.parseLong(value);
                return actualValue >= number;
            } catch (NumberFormatException e) {
                return false;
            }
        };
    }

    public static Predicate<String> isValidRegex() {
        return value -> {
            try {
                if (value == null)
                    return false;
                Pattern.compile(value);
                return true;
            } catch (PatternSyntaxException e) {
                return false;
            }
        };
    }

    public RuleBuilder(String fieldName) {
        this.fieldName = fieldName;
    }

    /**
     * Adds the value function, with the corresponding predicate to the checks list
     *
     * @param valueRetriever
     *         The value function to be used to retrieve value
     * @param predicate
     *         The predicate to be tested against the retrieved value from {@link Function}
     * @return Current builder instance
     */
    public <M> RuleBuilder<V> check(Function<V, M> valueRetriever, Predicate<M> predicate) {
        conditions.add(new Condition(predicate, valueRetriever));
        return this;
    }

    /**
     * Adds the value function, with the corresponding predicate to the checks list
     *
     * @param valueRetriever
     *         Predicate which will be used to check condition
     * @param expectedValue
     *         The value which will be compare in predicate
     * @return Current builder instance
     */
    public RuleBuilder<V> check(Predicate<V> valueRetriever, boolean expectedValue) {
        conditions.add(new Condition(arg -> arg == expectedValue, valueRetriever::test));
        return this;
    }

    /**
     * Sets the message supplier, which will be used to get the actual error message for the field error
     *
     * @param messageSupplier
     *         The supplier for the error message
     * @return Current builder's instance
     */
    public RuleBuilder<V> ifMatchesThenSet(Supplier<String> messageSupplier) {
        this.message = unused -> messageSupplier.get();
        return this;
    }

    /**
     * Sets the message function, which will be used to get the actual error message for the field error
     *
     * @param messageFunction the function, which will be used to get proper error message
     * @return Current builder's instance
     */
    public RuleBuilder<V> ifMatchesThenSet(Function<V, String> messageFunction) {
        this.message = messageFunction;
        return this;
    }

    /**
     * Build the {@link ValidationRule} with the supplied value functions and their corresponding predicates, all of
     * which must be positively tested to generate aa {@link FieldValidationError error}
     *
     * @return {@link ValidationRule} A rule that can be applied on the {@link V} entity to be validated
     */
    public ValidationRule<V> build() {
        return (arg) -> conditions.stream()
                .allMatch((condition -> condition.evaluate(arg))) ? FieldValidationError.of(fieldName, message.apply(arg)) : null;
    }
}
