package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.RemoteCrowdDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.CROWD_DIRECTORY.INVALID_APPLICATION_NAME;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.CROWD_DIRECTORY.INVALID_CROWD_SERVER_URL;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.CROWD_DIRECTORY.INVALID_HTTP_PROXY_PORT;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.CROWD_DIRECTORY.INVALID_PASSWORD;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.inLongRange;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.isNull;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.isValidURI;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.notNull;

public class RemoteCrowdConnectionValidator extends DirectoryValidator {

    public static final String CROWD_SERVER_URL = "url";
    public static final String APPLICATION_NAME = "applicationName";
    public static final String APPLICATION_PASSWORD = "applicationPassword";
    public static final String CROWD_HTTP_PROXY_PORT = "httpProxyPort";
    public static final Long MAX_PORT_NUMBER = 65535L;

    public RemoteCrowdConnectionValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();
        ruleListBuilder.add(ruleFor(CROWD_SERVER_URL)
                .check(valueOf(RemoteCrowdDirectory.CROWD_SERVER_URL), isNull().or(not(isValidURI())))
                .ifMatchesThenSet(message(i18nHelper, INVALID_CROWD_SERVER_URL)).build());

        ruleListBuilder.add(ruleFor(APPLICATION_NAME)
                .check(valueOf(RemoteCrowdDirectory.APPLICATION_NAME), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_APPLICATION_NAME)).build());

        ruleListBuilder.add(ruleFor(APPLICATION_PASSWORD)
                .check(valueOf(RemoteCrowdDirectory.APPLICATION_PASSWORD), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_PASSWORD)).build());

        ruleListBuilder.add(ruleFor(CROWD_HTTP_PROXY_PORT)
                .check(valueOf(RemoteCrowdDirectory.CROWD_HTTP_PROXY_PORT), notNull().and(not(inLongRange(0L, MAX_PORT_NUMBER))))
                .ifMatchesThenSet(message(i18nHelper, INVALID_HTTP_PROXY_PORT)).build());
        return ruleListBuilder.build();
    }
}
