package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.INVALID_PAGE_SIZE;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.LOCAL_GROUPS_ENABLED_WITHOUT_CACHE;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.LOCAL_USER_STATUS_WITHOUT_CACHE;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.greaterThanOrEquals;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;

public class LDAPConnectorValidator extends DirectoryValidator {

    public static final String PAGED_RESULTS_SIZE_KEY = "pagedResultsSize";
    public static final String POLLING_INTERVAL_IN_MIN_KEY = "pollingIntervalInMin";
    public static final String LOCAL_USER_STATUS_KEY = "localUserStatusEnabled";
    public static final String LOCAL_GROUPS_ENABLED = "localGroupsEnabled";

    public LDAPConnectorValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();
        ruleListBuilder.add(ruleFor(PAGED_RESULTS_SIZE_KEY)
                .check(valueOf(LDAPPropertiesMapper.LDAP_PAGEDRESULTS_KEY), eq(Boolean.TRUE.toString()))
                .check(valueOf(LDAPPropertiesMapper.LDAP_PAGEDRESULTS_SIZE), not(greaterThanOrEquals(100L)))
                .ifMatchesThenSet(message(i18nHelper, INVALID_PAGE_SIZE)).build());

        ruleListBuilder.add(ruleFor(LOCAL_USER_STATUS_KEY)
                .check(valueOf(DirectoryImpl.ATTRIBUTE_KEY_LOCAL_USER_STATUS), eq(Boolean.TRUE.toString()))
                .check(valueOf(DirectoryProperties.CACHE_ENABLED), eq(Boolean.FALSE.toString()))
                .ifMatchesThenSet(message(i18nHelper, LOCAL_USER_STATUS_WITHOUT_CACHE))
                .build());

        ruleListBuilder.add(ruleFor(LOCAL_GROUPS_ENABLED)
                .check(valueOf(LDAPPropertiesMapper.LOCAL_GROUPS), eq(Boolean.TRUE.toString()))
                .check(valueOf(DirectoryProperties.CACHE_ENABLED), eq(Boolean.FALSE.toString()))
                .ifMatchesThenSet(message(i18nHelper, LOCAL_GROUPS_ENABLED_WITHOUT_CACHE))
                .build());

        return ruleListBuilder.build();
    }
}
