package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.DelegatedAuthenticationDirectory;
import com.atlassian.crowd.directory.GenericLDAP;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.EMPTY_BASE_DN;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.INVALID_BASE_DN;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.LDAP_CONNECTION.INVALID_CONNECTOR_URL;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.matchesAll;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.regex;

public class DelegatedDirectoryConnectionValidator extends DirectoryValidator {

    public static final String BASE_DN_KEY = "baseDN";
    public static final String LDAP_URL_KEY = "URL";
    private static final String VALID_DN_PATTERN = ".+=.+";

    public DelegatedDirectoryConnectionValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();

        ruleListBuilder.add(ruleFor(LDAP_URL_KEY)
                .check(valueOf(LDAPPropertiesMapper.LDAP_URL_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_CONNECTOR_URL)).build());

        ruleListBuilder.add(ruleFor(BASE_DN_KEY)
                .check(valueOf(DelegatedAuthenticationDirectory.ATTRIBUTE_LDAP_DIRECTORY_CLASS), not(eq(GenericLDAP.class.getCanonicalName())))
                .check(valueOf(LDAPPropertiesMapper.LDAP_BASEDN_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, EMPTY_BASE_DN)).build());

        ruleListBuilder.add(ruleFor(BASE_DN_KEY)
                .check(valueOf(DelegatedAuthenticationDirectory.ATTRIBUTE_LDAP_DIRECTORY_CLASS), not(eq(GenericLDAP.class.getCanonicalName())))
                .check(valueOf(LDAPPropertiesMapper.LDAP_BASEDN_KEY), matchesAll(StringUtils::isNotBlank, not(regex(VALID_DN_PATTERN))))
                .ifMatchesThenSet(message(i18nHelper, INVALID_BASE_DN)).build());

        return ruleListBuilder.build();
    }
}
