package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.AzureAdDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.AZURE_AD;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.greaterThanOrEquals;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.isValidURI;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;

public class AzureADConnectorValidator extends DirectoryValidator {

    public static final String GRAPH_ENDPOINT_FIELD = "graphEndpoint";
    public static final String AUTHORITY_ENDPOINT_FIELD = "authorityEndpoint";
    public static final String WEB_APP_ID_FIELD = "webAppId";
    public static final String WEB_APP_SECRET_FIELD = "webAppSecret";
    public static final String NATIVE_APP_ID = "nativeAppId";
    public static final String TENANT_ID_FIELD = "tenantId";
    public static final String READ_TIMEOUT_IN_SEC = "readTimeoutInSec";
    public static final String CONNECTION_TIMEOUT_IN_SEC = "connectionTimeoutInSec";

    public AzureADConnectorValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();

        ruleListBuilder.add(ruleFor(AUTHORITY_ENDPOINT_FIELD)
                .check(valueOf(AzureAdDirectory.REGION_ATTRIBUTE), eq(AzureAdDirectory.CUSTOM_REGION_ATTRIBUTE_VALUE))
                .check(valueOf(AzureAdDirectory.AUTHORITY_API_ENDPOINT_ATTRIBUTE), not(isValidURI()))
                .ifMatchesThenSet(message(i18nHelper, AZURE_AD.INVALID_AUTHORITY_API))
                .build());

        ruleListBuilder.add(ruleFor(NATIVE_APP_ID)
                .check(valueOf(AzureAdDirectory.NATIVE_APP_ID_ATTRIBUTE), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, AZURE_AD.INVALID_NATIVE_APP_ID))
                .build());

        ruleListBuilder.add(ruleFor(READ_TIMEOUT_IN_SEC)
                .check(valueOf(SynchronisableDirectoryProperties.READ_TIMEOUT_IN_MILLISECONDS), not(greaterThanOrEquals(0L)))
                .ifMatchesThenSet(message(i18nHelper, AZURE_AD.INVALID_READ_TIMEOUT))
                .build());

        ruleListBuilder.add(ruleFor(CONNECTION_TIMEOUT_IN_SEC)
                .check(valueOf(SynchronisableDirectoryProperties.CONNECTION_TIMEOUT_IN_MILLISECONDS), not(greaterThanOrEquals(0L)))
                .ifMatchesThenSet(message(i18nHelper, AZURE_AD.INVALID_CONNECTION_TIMEOUT))
                .build());
        return ruleListBuilder.build();
    }
}
