package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.AzureAdDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.isValidURI;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.matchesAny;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;

public class AzureADConnectionValidator extends DirectoryValidator {

    public static final String GRAPH_ENDPOINT_FIELD = "graphEndpoint";
    public static final String WEB_APP_ID_FIELD = "webAppId";
    public static final String WEB_APP_SECRET_FIELD = "webAppSecret";
    public static final String TENANT_ID_FIELD = "tenantId";

    public AzureADConnectionValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();

        ruleListBuilder.add(ruleFor(TENANT_ID_FIELD)
                .check(valueOf(AzureAdDirectory.TENANT_ID_ATTRIBUTE), matchesAny(StringUtils::isBlank, StringUtils::containsWhitespace))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.AZURE_AD.INVALID_TENANT_ID))
                .build());

        ruleListBuilder.add(ruleFor(GRAPH_ENDPOINT_FIELD)
                .check(valueOf(AzureAdDirectory.REGION_ATTRIBUTE), eq(AzureAdDirectory.CUSTOM_REGION_ATTRIBUTE_VALUE))
                .check(valueOf(AzureAdDirectory.GRAPH_API_ENDPOINT_ATTRIBUTE), not(isValidURI()))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.AZURE_AD.INVALID_GRAPH_API))
                .build());

        ruleListBuilder.add(ruleFor(WEB_APP_ID_FIELD)
                .check(valueOf(AzureAdDirectory.WEBAPP_CLIENT_ID_ATTRIBUTE), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.AZURE_AD.INVALID_WEBAPP_ID))
                .build());

        ruleListBuilder.add(ruleFor(WEB_APP_SECRET_FIELD)
                .check(valueOf(AzureAdDirectory.WEBAPP_CLIENT_SECRET_ATTRIBUTE), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.AZURE_AD.INVALID_WEBAPP_SECRET))
                .build());

        return ruleListBuilder.build();
    }
}
