package com.atlassian.crowd.embedded.validator;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.util.I18nHelper;
import com.atlassian.crowd.validator.ValidationError;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Represents Validator which validates {@link Directory} implementations
 *
 * @since 3.2.0
 */
public abstract class DirectoryValidator implements Validator<Directory> {

    private final List<ValidationRule<Directory>> validationRules;

    public DirectoryValidator(I18nHelper i18nHelper) {
        validationRules = initializeValidators(i18nHelper);
    }

    /**
     * Validates the Directory, against the rules supplied by the implementor of the interface
     *
     * @param entity
     *         Entity to be validated
     * @return {@link List} list of validation errors generated
     */
    @Override
    public List<ValidationError> validate(Directory entity) {
        return validationRules.stream()
                .map(rule -> rule.apply(entity))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * Returns the list of validation rules to be used for validating the {@link Directory}
     *
     * @param i18nHelper
     *         instance of i18helper to get the message from supplied keys
     * @return list of validation rules
     */
    protected abstract List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper);
}
