package com.atlassian.crowd.embedded.core;

import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.crowd.embedded.api.GroupWithAttributes;
import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.api.UserCapabilities;
import com.atlassian.crowd.embedded.api.UserWithAttributes;
import com.atlassian.crowd.exception.FailedAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.exception.embedded.InvalidGroupException;
import com.atlassian.crowd.exception.runtime.GroupNotFoundException;
import com.atlassian.crowd.exception.runtime.OperationFailedException;
import com.atlassian.crowd.exception.runtime.UserNotFoundException;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;

import java.util.Set;

/**
 * Abstract implementation of CrowdService which simply delegates all the operations.
 */
abstract class DelegatingCrowdService implements CrowdService {
    protected final CrowdService crowdService;

    public DelegatingCrowdService(CrowdService crowdService) {
        this.crowdService = crowdService;
    }

    @Override
    public User authenticate(String name, String credential) throws FailedAuthenticationException, OperationFailedException {
        return crowdService.authenticate(name, credential);
    }

    @Override
    public User getRemoteUser(String username) throws OperationFailedException {
        return crowdService.getRemoteUser(username);
    }

    @Override
    public User getUser(String name) {
        return crowdService.getUser(name);
    }

    @Override
    public UserWithAttributes getUserWithAttributes(String name) {
        return crowdService.getUserWithAttributes(name);
    }

    @Override
    public Group getGroup(String name) {
        return crowdService.getGroup(name);
    }

    @Override
    public GroupWithAttributes getGroupWithAttributes(String name) {
        return crowdService.getGroupWithAttributes(name);
    }

    @Override
    public <T> Iterable<T> search(Query<T> query) {
        return crowdService.search(query);
    }

    @Override
    public boolean isUserMemberOfGroup(String userName, String groupName) {
        return crowdService.isUserMemberOfGroup(userName, groupName);
    }

    @Override
    public boolean isUserMemberOfGroup(User user, Group group) {
        return crowdService.isUserMemberOfGroup(user, group);
    }

    @Override
    public boolean isGroupMemberOfGroup(String childGroupName, String parentGroupName) {
        return crowdService.isGroupMemberOfGroup(childGroupName, parentGroupName);
    }

    @Override
    public boolean isGroupMemberOfGroup(Group childGroup, Group parentGroup) {
        return crowdService.isGroupMemberOfGroup(childGroup, parentGroup);
    }

    @Override
    public UserWithAttributes addUser(UserWithAttributes user, String credential) throws InvalidUserException, InvalidCredentialException, OperationNotPermittedException, OperationFailedException {
        return crowdService.addUser(user, credential);
    }

    @Override
    public User addUser(User user, String credential) throws InvalidUserException, InvalidCredentialException, OperationNotPermittedException, OperationFailedException {
        return addUser(UserTemplateWithAttributes.toUserWithNoAttributes(user), credential);
    }

    @Override
    public User updateUser(User user) throws UserNotFoundException, InvalidUserException, OperationNotPermittedException, OperationFailedException {
        return crowdService.updateUser(user);
    }

    @Override
    public User renameUser(User user, String newUsername) throws UserNotFoundException, InvalidUserException, OperationNotPermittedException, OperationFailedException {
        return crowdService.renameUser(user, newUsername);
    }

    @Override
    public void updateUserCredential(User user, String credential) throws UserNotFoundException, InvalidCredentialException, OperationNotPermittedException, OperationFailedException {
        crowdService.updateUserCredential(user, credential);
    }

    @Override
    public void setUserAttribute(User user, String attributeName, String attributeValue) throws UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.setUserAttribute(user, attributeName, attributeValue);
    }

    @Override
    public void setUserAttribute(User user, String attributeName, Set<String> attributeValues) throws UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.setUserAttribute(user, attributeName, attributeValues);
    }

    @Override
    public void removeUserAttribute(User user, String attributeName) throws UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.removeUserAttribute(user, attributeName);
    }

    @Override
    public void removeAllUserAttributes(User user) throws UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.removeAllUserAttributes(user);
    }

    @Override
    public boolean removeUser(User user) throws OperationNotPermittedException, OperationFailedException {
        return crowdService.removeUser(user);
    }

    @Override
    public Group addGroup(Group group) throws InvalidGroupException, OperationNotPermittedException, OperationFailedException {
        return crowdService.addGroup(group);
    }

    @Override
    public Group updateGroup(Group group) throws GroupNotFoundException, InvalidGroupException, OperationNotPermittedException, OperationFailedException {
        return crowdService.updateGroup(group);
    }

    @Override
    public void setGroupAttribute(Group group, String attributeName, String attributeValue) throws GroupNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.setGroupAttribute(group, attributeName, attributeValue);
    }

    @Override
    public void setGroupAttribute(Group group, String attributeName, Set<String> attributeValues) throws GroupNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.setGroupAttribute(group, attributeName, attributeValues);
    }

    @Override
    public void removeGroupAttribute(Group group, String attributeName) throws GroupNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.removeGroupAttribute(group, attributeName);
    }

    @Override
    public void removeAllGroupAttributes(Group group) throws GroupNotFoundException, OperationNotPermittedException, OperationFailedException {
        crowdService.removeAllGroupAttributes(group);
    }

    @Override
    public boolean removeGroup(Group group) throws OperationNotPermittedException, OperationFailedException {
        return crowdService.removeGroup(group);
    }

    @Override
    public boolean addUserToGroup(User user, Group group) throws GroupNotFoundException, UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        return crowdService.addUserToGroup(user, group);
    }

    @Override
    public boolean addGroupToGroup(Group childGroup, Group parentGroup) throws GroupNotFoundException, OperationNotPermittedException, InvalidMembershipException, OperationFailedException {
        return crowdService.addGroupToGroup(childGroup, parentGroup);
    }

    @Override
    public boolean removeUserFromGroup(User user, Group group) throws GroupNotFoundException, UserNotFoundException, OperationNotPermittedException, OperationFailedException {
        return crowdService.removeUserFromGroup(user, group);
    }

    @Override
    public boolean removeGroupFromGroup(Group childGroup, Group parentGroup) throws GroupNotFoundException, OperationNotPermittedException, OperationFailedException {
        return crowdService.removeGroupFromGroup(childGroup, parentGroup);
    }

    @Override
    public boolean isUserDirectGroupMember(User user, Group group) throws OperationFailedException {
        return crowdService.isUserDirectGroupMember(user, group);
    }

    @Override
    public boolean isGroupDirectGroupMember(Group childGroup, Group parentGroup) throws OperationFailedException {
        return crowdService.isGroupDirectGroupMember(childGroup, parentGroup);
    }

    @Override
    public UserCapabilities getCapabilitiesForNewUsers() {
        return crowdService.getCapabilitiesForNewUsers();
    }
}
