package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;

import java.util.List;

import static com.atlassian.crowd.directory.SyncScheduleType.CRON_EXPRESSION;
import static com.atlassian.crowd.directory.SynchronisableDirectoryProperties.CACHE_SYNCHRONISATION_TYPE;
import static com.atlassian.crowd.directory.SynchronisableDirectoryProperties.CACHE_SYNCHRONISE_CRON;
import static com.atlassian.crowd.directory.SynchronisableDirectoryProperties.CACHE_SYNCHRONISE_INTERVAL;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.SYNCHRONISATION_CONFIGURATION.INVALID_CRON_EXPRESSION;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.SYNCHRONISATION_CONFIGURATION.INVALID_POLLING_INTERVAL;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.greaterThanOrEquals;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;

public class SynchronisationSchedulingConfigValidator extends DirectoryValidator {
    public static final String CRON_EXPRESSION_FIELD = "cronExpression";
    public static final String POLLING_INTERVAL_FIELD = "pollingIntervalInMin";

    public SynchronisationSchedulingConfigValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        return ImmutableList.of(
                ruleFor(POLLING_INTERVAL_FIELD)
                        .check(valueOf(CACHE_SYNCHRONISATION_TYPE), not(eq(CRON_EXPRESSION)))
                        .check(valueOf(CACHE_SYNCHRONISE_INTERVAL), not(greaterThanOrEquals(1L)))
                        .ifMatchesThenSet(message(i18nHelper, INVALID_POLLING_INTERVAL))
                        .build(),
                ruleFor(CRON_EXPRESSION_FIELD)
                        .check(valueOf(CACHE_SYNCHRONISATION_TYPE), eq(CRON_EXPRESSION))
                        .check(valueOf(CACHE_SYNCHRONISE_CRON), value -> !CrowdCronExpressionValidator.isValid(value))
                        .ifMatchesThenSet(message(i18nHelper, INVALID_CRON_EXPRESSION))
                        .build()
        );
    }
}
