package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.embedded.validator.DirectoryValidatorFactory;
import com.atlassian.crowd.embedded.validator.Validator;
import com.atlassian.crowd.util.I18nHelper;
import com.atlassian.crowd.validator.DirectoryValidationContext;
import com.google.common.collect.ImmutableList;

import java.util.EnumSet;
import java.util.List;
import java.util.stream.Collectors;

import static com.atlassian.crowd.validator.DirectoryValidationContext.DEFAULT;

public class DirectoryValidatorFactoryImpl implements DirectoryValidatorFactory {

    private final I18nHelper i18nHelper;

    public DirectoryValidatorFactoryImpl(I18nHelper i18nHelper) {
        this.i18nHelper = i18nHelper;
    }

    private Validator<Directory> getCrowdDirectoryValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new RemoteCrowdConnectionValidator(i18nHelper), new RemoteCrowdConnectorValidator(i18nHelper), new SynchronisationSchedulingConfigValidator(i18nHelper)));
            }
            case CONNECTION:
                return new RemoteCrowdConnectionValidator(i18nHelper);
            case CONNECTOR_ATTRIBUTES:
                return new RemoteCrowdConnectorValidator(i18nHelper);
            case SYNCHRONISATION_SCHEDULING:
                return new SynchronisationSchedulingConfigValidator(i18nHelper);
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.CROWD, context.name()));
        }
    }

    private Validator<Directory> getCustomDirectoryValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new CustomDirectoryValidator(i18nHelper)));
            }
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.CUSTOM, context.name()));
        }
    }

    private Validator<Directory> getAzureADValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new AzureADConnectionValidator(i18nHelper),
                        new AzureADConnectorValidator(i18nHelper), new SynchronisationSchedulingConfigValidator(i18nHelper)));
            }
            case CONNECTION:
                return new AzureADConnectionValidator(i18nHelper);
            case CONNECTOR_ATTRIBUTES:
                return new AzureADConnectorValidator(i18nHelper);
            case SYNCHRONISATION_SCHEDULING:
                return new SynchronisationSchedulingConfigValidator(i18nHelper);
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.AZURE_AD, context.name()));
        }
    }

    private Validator<Directory> getInternalDirectoryValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new InternalDirectoryValidator(i18nHelper)));
            }
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.INTERNAL, context.name()));
        }
    }

    private Validator<Directory> getDelegatedDirectoryValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new DelegatedDirectoryConnectionValidator(i18nHelper), new LDAPConnectorValidator(i18nHelper),
                        new LDAPUserConfigValidator(i18nHelper), new LDAPGroupConfigValidator(i18nHelper)));
            }
            case CONNECTION:
                return new DelegatedDirectoryConnectionValidator(i18nHelper);
            case CONNECTOR_ATTRIBUTES:
                return new LDAPConnectorValidator(i18nHelper);
            case USER_CONFIGURATION:
                return new LDAPUserConfigValidator(i18nHelper);
            case GROUP_CONFIGURATION:
                return new LDAPGroupConfigValidator(i18nHelper);
            case USER_SEARCH:
                return new LDAPUserSearchConfigValidator(i18nHelper);
            case GROUP_SEARCH:
                return new LDAPGroupSearchConfigValidator(i18nHelper);
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.DELEGATING, context.name()));
        }
    }

    private Validator<Directory> getConnectorValidator(DirectoryValidationContext context) {
        switch (context) {
            case DEFAULT: {
                return new CompoundValidator<>(ImmutableList.of(new LDAPConnectionValidator(i18nHelper), new LDAPConnectorValidator(i18nHelper), new SynchronisationSchedulingConfigValidator(i18nHelper),
                        new LDAPUserConfigValidator(i18nHelper), new LDAPGroupConfigValidator(i18nHelper)));
            }
            case CONNECTION:
                return new LDAPConnectionValidator(i18nHelper);
            case CONNECTOR_ATTRIBUTES:
                return new LDAPConnectorValidator(i18nHelper);
            case USER_CONFIGURATION:
                return new LDAPUserConfigValidator(i18nHelper);
            case GROUP_CONFIGURATION:
                return new LDAPGroupConfigValidator(i18nHelper);
            case USER_SEARCH:
                return new LDAPUserSearchConfigValidator(i18nHelper);
            case GROUP_SEARCH:
                return new LDAPGroupSearchConfigValidator(i18nHelper);
            case SYNCHRONISATION_SCHEDULING:
                return new SynchronisationSchedulingConfigValidator(i18nHelper);
            default:
                throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", DirectoryType.CONNECTOR, context.name()));
        }
    }

    private Validator<Directory> getContextualValidator(DirectoryType directoryType, DirectoryValidationContext validationContext) {
        if (directoryType == DirectoryType.INTERNAL) {
            return getInternalDirectoryValidator(validationContext);
        } else if (directoryType == DirectoryType.CONNECTOR) {
            return getConnectorValidator(validationContext);
        } else if (directoryType == DirectoryType.DELEGATING) {
            return getDelegatedDirectoryValidator(validationContext);
        } else if (directoryType == DirectoryType.CROWD) {
            return getCrowdDirectoryValidator(validationContext);
        } else if (directoryType == DirectoryType.CUSTOM) {
            return getCustomDirectoryValidator(validationContext);
        } else if (directoryType == DirectoryType.AZURE_AD) {
            return getAzureADValidator(validationContext);
        }

        throw new UnsupportedOperationException(String.format("No validator is available for type: %s, and context: %s", directoryType.name(), validationContext.name()));
    }

    @Override
    public Validator<Directory> getValidator(DirectoryType directoryType, EnumSet<DirectoryValidationContext> directoryValidationContexts) {
        // If DEFAULT context is required, in order to prevent un-necessary nesting of compound validators
        if (directoryValidationContexts.contains(DEFAULT)) {
            return getContextualValidator(directoryType, DEFAULT);
        }

        List<Validator<Directory>> directoryValidators = directoryValidationContexts.stream()
                .filter(context -> context != DirectoryValidationContext.DEFAULT)
                .map(context -> getContextualValidator(directoryType, context))
                .collect(Collectors.toList());

        return new CompoundValidator<>(directoryValidators);
    }
}
