package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.DelegatedAuthenticationDirectory;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USERNAME;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USERNAME_RDN;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_DISPLAY_NAME;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_FIRST_NAME;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_GROUP;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_LAST_NAME;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_MAIL;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_OBJECT_CLASS;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_OBJECT_FILTER;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.USER_CONFIGURATION.INVALID_USER_PASSWORD;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.eq;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;

public class LDAPUserConfigValidator extends DirectoryValidator {

    public static final String USER_FIRST_NAME_ATTRIBUTE = "userFirstnameAttr";
    public static final String USER_GROUP_MEMBER_ATTRIBUTE = "userGroupMemberAttr";
    public static final String USER_LAST_NAME_ATTRIBUTE = "userLastnameAttr";
    public static final String USER_DISPLAY_NAME_ATTRIBUTE = "userDisplayNameAttr";
    public static final String USER_MAIL_ATTRIBUTE = "userMailAttr";
    public static final String USER_NAME_ATTRIBUTE = "userNameAttr";
    public static final String USERNAME_RDN_ATTRIBUTE = "userNameRdnAttr";
    public static final String USER_OBJECT_FILTER = "userObjectFilter";
    public static final String USER_OBJECT_CLASS = "userObjectClass";
    public static final String USER_PASSWORD_ATTRIBUTE = "userPasswordAttr";

    public LDAPUserConfigValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();

        ruleListBuilder.add(ruleFor(USER_OBJECT_CLASS)
                .check(valueOf(LDAPPropertiesMapper.USER_OBJECTCLASS_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_OBJECT_CLASS))
                .build());

        ruleListBuilder.add(ruleFor(USER_OBJECT_FILTER)
                .check(valueOf(LDAPPropertiesMapper.USER_OBJECTFILTER_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_OBJECT_FILTER))
                .build());

        ruleListBuilder.add(ruleFor(USER_NAME_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_USERNAME_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USERNAME))
                .build());

        ruleListBuilder.add(ruleFor(USERNAME_RDN_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_USERNAME_RDN_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USERNAME_RDN))
                .build());

        ruleListBuilder.add(ruleFor(USER_FIRST_NAME_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_FIRSTNAME_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_FIRST_NAME))
                .build());

        ruleListBuilder.add(ruleFor(USER_LAST_NAME_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_LASTNAME_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_LAST_NAME))
                .build());

        ruleListBuilder.add(ruleFor(USER_DISPLAY_NAME_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_DISPLAYNAME_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_DISPLAY_NAME))
                .build());

        ruleListBuilder.add(ruleFor(USER_MAIL_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_EMAIL_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_MAIL))
                .build());

        ruleListBuilder.add(ruleFor(USER_GROUP_MEMBER_ATTRIBUTE)
                .check(valueOf(LDAPPropertiesMapper.USER_GROUP_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_GROUP))
                .build());

        ruleListBuilder.add(ruleFor(USER_PASSWORD_ATTRIBUTE)
                .check(Directory::getImplementationClass, not(eq(DelegatedAuthenticationDirectory.class.getName())))
                .check(valueOf(LDAPPropertiesMapper.USER_PASSWORD_KEY), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, INVALID_USER_PASSWORD))
                .build());

        return ruleListBuilder.build();
    }
}
