package com.atlassian.crowd.embedded.validator.impl;

import com.atlassian.crowd.directory.AbstractInternalDirectory;
import com.atlassian.crowd.directory.InternalDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.validator.DirectoryValidator;
import com.atlassian.crowd.embedded.validator.ValidationRule;
import com.atlassian.crowd.util.I18nHelper;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Longs;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_REMIND_PERIODS_FORMAT;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_REMIND_PERIODS_VALUES;
import static com.atlassian.crowd.embedded.validator.impl.DirectoryValidatorMessages.INTERNAL_DIRECTORY.MAX_PASSWORD_CHANGE_TIME_TOO_LOW;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.ruleFor;
import static com.atlassian.crowd.embedded.validator.rule.DirectoryRuleBuilder.valueOf;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.greaterThanOrEquals;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.isValidRegex;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.message;
import static com.atlassian.crowd.embedded.validator.rule.RuleBuilder.not;


public class InternalDirectoryValidator extends DirectoryValidator {

    public static final String PASSWORD_REGEX = "passwordRegex";
    public static final String PASSWORD_MAX_CHANGE_TIME = "passwordMaxChangeTime";
    public static final String PASSWORD_MAX_AUTHENTICATE_ATTEMPTS = "passwordMaxAttempts";
    public static final String PASSWORD_HISTORY_COUNT = "passwordHistoryCount";
    public static final String USER_ENCRYPTION_METHOD = "userEncryptionMethod";
    public static final String REMIND_PERIODS = "passwordExpirationNotificationPeriods";
    private static final int MINIMUM_PASSWORD_MAX_CHANGE_TIME_TO_SET_REMINDERS = 3;

    public InternalDirectoryValidator(I18nHelper i18nHelper) {
        super(i18nHelper);
    }

    @Override
    protected List<ValidationRule<Directory>> initializeValidators(I18nHelper i18nHelper) {
        final ImmutableList.Builder<ValidationRule<Directory>> ruleListBuilder = ImmutableList.builder();

        ruleListBuilder.add(ruleFor(PASSWORD_REGEX)
                .check(valueOf(AbstractInternalDirectory.ATTRIBUTE_PASSWORD_REGEX), not(isValidRegex()))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_PASSWORD_REGEX_PATTERN)).build());

        ruleListBuilder.add(ruleFor(PASSWORD_MAX_AUTHENTICATE_ATTEMPTS)
                .check(valueOf(InternalDirectory.ATTRIBUTE_PASSWORD_MAX_ATTEMPTS), not(greaterThanOrEquals(0L)))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_PASSWORD_MAX_AUTHENTICATE_ATTEMPTS)).build());

        ruleListBuilder.add(ruleFor(PASSWORD_MAX_CHANGE_TIME)
                .check(valueOf(InternalDirectory.ATTRIBUTE_PASSWORD_MAX_CHANGE_TIME), not(greaterThanOrEquals(0L)))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_PASSWORD_MAX_CHANGE_TIME)).build());

        ruleListBuilder.add(ruleFor(PASSWORD_HISTORY_COUNT)
                .check(valueOf(InternalDirectory.ATTRIBUTE_PASSWORD_HISTORY_COUNT), not(greaterThanOrEquals(0L)))
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_PASSWORD_HISTORY_COUNTS)).build());

        ruleListBuilder.add(ruleFor(USER_ENCRYPTION_METHOD)
                .check(valueOf(InternalDirectory.ATTRIBUTE_USER_ENCRYPTION_METHOD), StringUtils::isBlank)
                .ifMatchesThenSet(message(i18nHelper, DirectoryValidatorMessages.INTERNAL_DIRECTORY.INVALID_USER_ENCRYPTION_METHOD)).build());

        ruleListBuilder.add(ruleFor(REMIND_PERIODS)
                .check(dir -> new PasswordExpirationAttributes(
                                dir.getValue(InternalDirectory.ATTRIBUTE_PASSWORD_EXPIRATION_NOTIFICATION_PERIODS),
                                dir.getValue(InternalDirectory.ATTRIBUTE_PASSWORD_MAX_CHANGE_TIME)),
                        not(this::validRemindPeriodsFormat))
                .ifMatchesThenSet(message(i18nHelper, INVALID_REMIND_PERIODS_FORMAT)).build());

        ruleListBuilder.add(ruleFor(REMIND_PERIODS)
                .check(dir -> new PasswordExpirationAttributes(
                                dir.getValue(InternalDirectory.ATTRIBUTE_PASSWORD_EXPIRATION_NOTIFICATION_PERIODS),
                                dir.getValue(InternalDirectory.ATTRIBUTE_PASSWORD_MAX_CHANGE_TIME)),
                        not(this::validRemindPeriodsValues))
                .ifMatchesThenSet(dir -> {
                    final int passwordMaxChangeTime = Integer.parseInt(dir.getValue(InternalDirectory.ATTRIBUTE_PASSWORD_MAX_CHANGE_TIME));
                    if (passwordMaxChangeTime < MINIMUM_PASSWORD_MAX_CHANGE_TIME_TO_SET_REMINDERS) {
                        return i18nHelper.getText(MAX_PASSWORD_CHANGE_TIME_TOO_LOW);
                    } else {
                        return i18nHelper.getText(INVALID_REMIND_PERIODS_VALUES, ImmutableList.of(passwordMaxChangeTime, passwordMaxChangeTime - 1));
                    }
                }).build());

        return ruleListBuilder.build();
    }

    private boolean validRemindPeriodsValues(PasswordExpirationAttributes attributes) {
        if (attributes.getRemindPeriods().size() == 0 || attributes.getRemindPeriods().contains(null)) {
            return true;
        }

        final TreeSet<Long> remindPeriods = new TreeSet(Comparator.reverseOrder());
        remindPeriods.addAll(attributes.getRemindPeriods());
        final Long passwordMaxChangeTime = Longs.tryParse(attributes.getPasswordMaxChangeTime());
        return remindPeriods.size() > 0
                && remindPeriods.last() > 0
                && remindPeriods.first() < passwordMaxChangeTime
                && passwordMaxChangeTime >= MINIMUM_PASSWORD_MAX_CHANGE_TIME_TO_SET_REMINDERS;
    }

    private boolean validRemindPeriodsFormat(PasswordExpirationAttributes attributes) {
        final Set<Long> uniqueRemindPeriods = new HashSet<>(attributes.getRemindPeriods());

        return uniqueRemindPeriods.size() == attributes.getRemindPeriods().size() && !attributes.getRemindPeriods().contains(null);
    }

    private static class PasswordExpirationAttributes {
        private final String passwordMaxChangeTime;
        private final List<Long> remindPeriodsAsArray;

        public PasswordExpirationAttributes(String remindPeriods, String passwordMaxChangeTime) {
            this.remindPeriodsAsArray = parseRemindPeriodsAsArray(remindPeriods);
            this.passwordMaxChangeTime = passwordMaxChangeTime;
        }

        public String getPasswordMaxChangeTime() {
            return passwordMaxChangeTime;
        }

        public List<Long> getRemindPeriods() {
            return remindPeriodsAsArray;
        }

        protected List<Long> parseRemindPeriodsAsArray(String remindPeriods) {
            return remindPeriods.equals("")
                    ? Collections.emptyList()
                    : Arrays.stream(remindPeriods.split(","))
                        .map(String::trim)
                        .map(Longs::tryParse)
                        .collect(Collectors.toList());
        }
    }
}
