package com.atlassian.crowd.embedded.impl;

import com.google.common.collect.ForwardingMap;
import com.google.common.collect.Maps;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

/**
 * This class behaves like a HashMap with lower-case String keys. All key
 * arguments are lower-cased before further processing.
 *
 * @param <V> value type
 */
public class IdentifierMap<V> extends ForwardingMap<String, V> {
    private final HashMap<String, V> delegate;

    public IdentifierMap() {
        delegate = Maps.newHashMap();
    }

    public IdentifierMap(int expectedSize) {
        delegate = Maps.newHashMapWithExpectedSize(expectedSize);
    }

    public IdentifierMap(Map<String, V> map) {
        this(map.size());
        putAll(map);
    }

    @Override
    protected Map<String, V> delegate() {
        return delegate;
    }

    private Object lowercase(Object key) {
        return key instanceof String ? IdentifierUtils.toLowerCase((String) key) : key;
    }

    @Override
    public V remove(Object key) {
        return delegate().remove(lowercase(key));
    }

    @Override
    public boolean containsKey(Object key) {
        return delegate().containsKey(lowercase(key));
    }

    @Override
    public V get(Object key) {
        return delegate().get(lowercase(key));
    }

    @Override
    public V put(String key, V value) {
        return delegate().put(IdentifierUtils.toLowerCase(key), value);
    }

    @Override
    public void putAll(Map<? extends String, ? extends V> map) {
        delegate().putAll(lowercaseMap(map));
    }

    private static <V> Map<? extends String, ? extends V> lowercaseMap(Map<? extends String, ? extends V> map) {
        if (map instanceof IdentifierMap) {
            return map;
        }
        Map<String, V> lowercaseMap = Maps.newHashMapWithExpectedSize(map.size());
        for (Entry<? extends String, ? extends V> entry : map.entrySet()) {
            lowercaseMap.put(IdentifierUtils.toLowerCase(entry.getKey()), entry.getValue());
        }
        return lowercaseMap;
    }

    public static <T> IdentifierMap<T> index(Collection<? extends T> items, Function<T, String> nameGetter) {
        IdentifierMap<T> result = new IdentifierMap<>(items.size());
        items.forEach(item -> result.put(nameGetter.apply(item), item));
        return result;
    }
}
