package com.atlassian.crowd.directory;

import com.atlassian.crowd.directory.synchronisation.utils.AddUpdateSets;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Individual methods that should be performed in transactions. No transactional integrity is
 * presumed or required between methods.
 */
public interface DirectoryCacheChangeOperations {
    // Operations mirrored from {@link DirectoryCache}.
    void deleteCachedUsersNotIn(Collection<? extends User> users, Date syncStartDate) throws OperationFailedException;

    void deleteCachedUsersByGuid(Set<String> usernames) throws OperationFailedException;

    void deleteCachedGroupsNotIn(GroupType groupType, List<? extends Group> ldapGroups, Date syncStartDate) throws OperationFailedException;

    void deleteCachedGroupsNotInByExternalId(Collection<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException;

    void deleteCachedGroups(Set<String> groupnames) throws OperationFailedException;

    void deleteCachedGroupsByGuids(Set<String> guids) throws OperationFailedException;

    // For addOrUpdateCachedUsers
    AddUpdateSets<UserTemplateWithCredentialAndAttributes, UserTemplate> getUsersToAddAndUpdate(
            Collection<? extends User> remoteUsers, Date syncStartDate) throws OperationFailedException;

    void addUsers(Set<UserTemplateWithCredentialAndAttributes> usersToAdd) throws OperationFailedException;

    void updateUsers(Collection<UserTemplate> usersToUpdate) throws OperationFailedException;

    // For addOrUpdateCachedGroups
    GroupsToAddUpdateReplace findGroupsToUpdate(Collection<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException;

    void removeGroups(Collection<String> keySet) throws OperationFailedException;

    void addGroups(Set<GroupTemplate> allToAdd) throws OperationFailedException;

    void updateGroups(Collection<GroupTemplate> groupsToUpdate) throws OperationFailedException;

    GroupShadowingType isGroupShadowed(Group group) throws OperationFailedException;

    // For syncUserMembershipsForGroup
    AddRemoveSets<String> findUserMembershipForGroupChanges(Group group, Collection<String> remoteUsers) throws OperationFailedException;

    void addUserMembershipsForGroup(Group group, Set<String> toAdd) throws OperationFailedException;

    void removeUserMembershipsForGroup(Group group, Set<String> toRemove) throws OperationFailedException;

    // For syncGroupMembershipsForGroup
    AddRemoveSets<String> findGroupMembershipForGroupChanges(Group parentGroup, Collection<String> remoteGroups) throws OperationFailedException;

    void addGroupMembershipsForGroup(Group parentGroup, Collection<String> toAdd) throws OperationFailedException;

    void removeGroupMembershipsForGroup(Group parentGroup, Collection<String> toRemove) throws OperationFailedException;

    // for inc user sync
    Set<String> getAllUserGuids() throws OperationFailedException;

    long getUserCount() throws OperationFailedException;

    // for inc group sync
    Set<String> getAllGroupGuids() throws OperationFailedException;

    long getGroupCount() throws OperationFailedException;

    long getExternalCachedGroupCount() throws OperationFailedException;

    Set<String> getAllLocalGroupNames() throws OperationFailedException;

    // -----------------------------------------------------------------------------------------------------------------
    // Event operations
    // -----------------------------------------------------------------------------------------------------------------

    void addOrUpdateCachedUser(User user) throws OperationFailedException;

    void deleteCachedUser(String username) throws OperationFailedException;

    void addOrUpdateCachedGroup(Group group) throws OperationFailedException;

    void deleteCachedGroup(String groupName) throws OperationFailedException;

    void addUserToGroup(String username, String groupName) throws OperationFailedException;

    void removeUserFromGroup(String username, String groupName) throws OperationFailedException;

    void addGroupToGroup(String childGroup, String parentGroup) throws OperationFailedException;

    void removeGroupFromGroup(String childGroup, String parentGroup) throws OperationFailedException;

    void syncGroupMembershipsForUser(String childUsername, Set<String> parentGroupNames) throws OperationFailedException;

    void syncGroupMembershipsAndMembersForGroup(String groupName, Set<String> parentGroupNames, Set<String> childGroupNames) throws OperationFailedException;

    UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException;

    GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException;

    Map<String, String> findUsersByExternalIds(Set<String> externalIds);

    /**
     * Searches the specified directory for group names of groups with the specified external ids
     *
     * @param externalIds external ids of the groups to find
     * @return a map from external ids to group names
     */
    Map<String, String> findGroupsByExternalIds(Set<String> externalIds) throws OperationFailedException;

    /**
     * Searches the specified directory for externalIds of groups with the specified names
     *
     * @param groupNames names of the groups to find
     * @return a map from group names to external ids
     */
    Map<String, String> findGroupsExternalIdsByNames(Set<String> groupNames) throws OperationFailedException;

    /**
     * Removes any syncing user attributes listed in deletedAttributes, and store any that are listed in storedAttributes
     *
     * If any non-syncing attributes are included, they will be ignored
     *
     * @param userName          user to apply attributes to
     * @param deletedAttributes attribute keys that need to be deleted
     * @param storedAttributes  attributes and values that need to be saved (will overwrite any existing attributes with the same keys)
     */
    void applySyncingUserAttributes(String userName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes)
            throws UserNotFoundException, OperationFailedException;

    /**
     * Removes any syncing group attributes listed in deletedAttributes, and store any that are listed in storedAttributes
     *
     * If any non-syncing attributes are included, they will be ignored
     *
     * @param groupName         group to apply attributes to
     * @param deletedAttributes attribute keys that need to be deleted
     * @param storedAttributes  attributes and values that need to be saved (will overwrite any existing attributes with the same keys)
     */
    void applySyncingGroupAttributes(String groupName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes) throws GroupNotFoundException, OperationFailedException;

    class GroupsToAddUpdateReplace {
        final Set<GroupTemplate> groupsToAdd, groupsToUpdate;
        final Map<String, GroupTemplate> groupsToReplace;

        public GroupsToAddUpdateReplace(Set<GroupTemplate> groupsToAdd, Set<GroupTemplate> groupsToUpdate,
                                 Map<String, GroupTemplate> groupsToReplace) {
            this.groupsToAdd = Collections.unmodifiableSet(groupsToAdd);
            this.groupsToUpdate = Collections.unmodifiableSet(groupsToUpdate);
            this.groupsToReplace = Collections.unmodifiableMap(groupsToReplace);
        }

        public Set<GroupTemplate> getGroupsToAdd() {
            return groupsToAdd;
        }

        public Set<GroupTemplate> getGroupsToUpdate() {
            return groupsToUpdate;
        }

        public Map<String, GroupTemplate> getGroupsToReplace() {
            return groupsToReplace;
        }
    }

    class AddRemoveSets<T> {
        final Set<T> toAdd;
        final Set<T> toRemove;

        public AddRemoveSets(Set<T> toAdd, Set<T> toRemove) {
            this.toAdd = Collections.unmodifiableSet(toAdd);
            this.toRemove = Collections.unmodifiableSet(toRemove);
        }
    }

    enum GroupShadowingType {
        SHADOWED_BY_ROLE,
        SHADOWED_BY_LOCAL_GROUP,
        NOT_SHADOWED,
        GROUP_REMOVED
    }
}
