package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.directory.synchronisation.CacheSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.PartialSynchronisationResult;
import com.atlassian.crowd.embedded.api.Attributes;
import com.atlassian.crowd.embedded.impl.IdentifierMap;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.util.TimedOperation;
import com.atlassian.crowd.util.TimedProgressOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.crowd.attribute.AttributePredicates.SYNCING_ATTRIBUTE;

/**
 * @since v2.1
 */
public abstract class AbstractCacheRefresher<G extends GroupWithAttributes> implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(AbstractCacheRefresher.class);

    protected final RemoteDirectory remoteDirectory;

    public AbstractCacheRefresher(final RemoteDirectory remoteDirectory) {
        this.remoteDirectory = remoteDirectory;
    }

    @Override
    public CacheSynchronisationResult synchroniseAll(DirectoryCache directoryCache) throws OperationFailedException {
        final PartialSynchronisationResult<? extends UserWithAttributes> allUsers = synchroniseAllUsers(directoryCache);
        if (isUserAttributeSynchronisationEnabled()) {
            synchroniseAllUserAttributes(allUsers.getResults(), directoryCache);
        }

        final PartialSynchronisationResult<G> allGroups = synchroniseAllGroups(directoryCache);
        if (isGroupAttributeSynchronisationEnabled()) {
            synchroniseAllGroupAttributes(allGroups.getResults(), directoryCache);
        }

        // We have to finish adding new Groups BEFORE we can add Nested Group memberships.
        synchroniseMemberships(allGroups.getResults(), directoryCache, true);
        return new CacheSynchronisationResult(true, null);
    }

    /**
     * This method is expected to fetch the users to update (or all currently in the remote directory)
     * and update the directory cache (call the appropriate method for deleting and then the appropriate method for
     * updating/adding). This will only fetch user attributes when user attribute synchronisation is enabled.
     * User attributes will be updated if fetched in {@link AbstractCacheRefresher#synchroniseAll(DirectoryCache)} if the directory
     * allows that.
     *
     * @param directoryCache the cache which this method should update
     * @return list of users that were passed into the directory cache and an optional sync token if the directory has
     * separate sync tokens for users and groups
     * @throws OperationFailedException
     */
    protected abstract PartialSynchronisationResult<? extends UserWithAttributes> synchroniseAllUsers(final DirectoryCache directoryCache) throws OperationFailedException;

    /**
     * This method is expected to fetch the groups to update (or all currently in the remote directory)
     * and update the directory cache (call the appropriate method for deleting and then the appropriate method for
     * updating/adding). This will only fetch group attributes when group attribute synchronisation is enabled.
     * Group attributes will be updated if fetched in {@link AbstractCacheRefresher#synchroniseAll(DirectoryCache)} if the directory
     * allows that.
     *
     * @param directoryCache the cache which this method should update
     * @return list of groups that were passed into the directory cache and an optional sync token if the directory has
     * separate sync tokens for users and groups
     * @throws OperationFailedException
     */
    protected abstract PartialSynchronisationResult<G> synchroniseAllGroups(final DirectoryCache directoryCache) throws OperationFailedException;

    protected Iterable<Membership> getMemberships(Collection<G> groups, boolean isFullSync) throws OperationFailedException {
        return remoteDirectory.getMemberships();
    }

    protected boolean isUserAttributeSynchronisationEnabled() {
        return Boolean.parseBoolean(remoteDirectory.getValue(DirectoryImpl.ATTRIBUTE_KEY_USER_ATTRIBUTES_SYNC_ENABLED));
    }

    protected boolean isGroupAttributeSynchronisationEnabled() {
        return Boolean.parseBoolean(remoteDirectory.getValue(DirectoryImpl.ATTRIBUTE_KEY_GROUP_ATTRIBUTES_SYNC_ENABLED));
    }


    protected void synchroniseAllUserAttributes(final Collection<? extends UserWithAttributes> remoteUsers, final DirectoryCache directoryCache)
            throws OperationFailedException {
        TimedOperation userAttributeSyncOperation = new TimedOperation();
        int failureCount = 0;
        for (UserWithAttributes remoteUser : remoteUsers) {
            try {
                UserWithAttributes internalUserWithAttributes = directoryCache.findUserWithAttributesByName(remoteUser.getName());

                Set<String> attributesToDelete = getAttributesToDelete(remoteUser, internalUserWithAttributes);
                Map<String, Set<String>> attributesToStore = getAttributesToStore(remoteUser, internalUserWithAttributes);

                directoryCache.applySyncingUserAttributes(remoteUser.getName(), attributesToDelete, attributesToStore);
            } catch (UserNotFoundException e) {
                // the user may have been removed from the cache. The next sync will fix the problem
                failureCount++;
                log.debug("Could not synchronize user attributes for user [{}]. User was not found in the cache.", remoteUser.getName());
            }
        }

        log.info(userAttributeSyncOperation.complete("finished user attribute sync with " + failureCount + " failures"));
    }

    protected void synchroniseAllGroupAttributes(final Collection<G> remoteGroups, final DirectoryCache directoryCache)
            throws OperationFailedException {
        TimedOperation groupAttributeSyncOperation = new TimedOperation();
        int failureCount = 0;
        for (GroupWithAttributes remoteGroup : remoteGroups) {
            try {
                GroupWithAttributes internalGroupWithAttributes = directoryCache.findGroupWithAttributesByName(remoteGroup.getName());

                Set<String> attributesToDelete = getAttributesToDelete(remoteGroup, internalGroupWithAttributes);
                Map<String, Set<String>> attributesToStore = getAttributesToStore(remoteGroup, internalGroupWithAttributes);

                directoryCache.applySyncingGroupAttributes(remoteGroup.getName(), attributesToDelete, attributesToStore);
            } catch (GroupNotFoundException e) {
                // the group may have been removed from the cache. The next sync will fix the problem
                failureCount++;
                log.debug("Could not synchronize group attributes for group [{}]. Group was not found in the cache.", remoteGroup.getName());
            }
        }

        log.info(groupAttributeSyncOperation.complete("finished group attribute sync with " + failureCount + " failures"));
    }

    /**
     * @param newAttributes the new attributes of the entity
     * @param oldAttributes the old attributes of the entity
     * @return the synchronisable attributes that are no longer present
     */
    static Set<String> getAttributesToDelete(Attributes newAttributes, Attributes oldAttributes) {
        Set<String> newAttributeNames = newAttributes.getKeys(); // optimisation
        return oldAttributes.getKeys().stream()
                .filter(SYNCING_ATTRIBUTE)
                .filter(attrName -> !newAttributeNames.contains(attrName))
                .collect(Collectors.toSet());
    }

    /**
     * @param newAttributes the new attributes of the entity
     * @param oldAttributes the old attributes of the entity
     * @return the synchronisable attributes that have changed value, including those which are new
     */
    static Map<String, Set<String>> getAttributesToStore(Attributes newAttributes, Attributes oldAttributes) {
        return newAttributes.getKeys().stream()
                .filter(SYNCING_ATTRIBUTE)
                .filter(attrName -> !Objects.equals(newAttributes.getValues(attrName), oldAttributes.getValues(attrName)))
                .collect(Collectors.toMap(attrName -> attrName, newAttributes::getValues));
    }

    protected void synchroniseMemberships(final Collection<G> remoteGroups, final DirectoryCache directoryCache, boolean isFullSync)
            throws OperationFailedException {
        if (log.isDebugEnabled()) {
            log.debug("Updating memberships for " + remoteGroups.size() + " groups from " + directoryDescription());
        }

        int total = remoteGroups.size();

        Map<String, G> groupsByName = new IdentifierMap<>();

        for (G g : remoteGroups) {
            String name = g.getName();
            if (null != groupsByName.put(name, g)) {
                throw new OperationFailedException("Unable to synchronise directory: duplicate groups with name '" + name + "'");
            }
        }

        TimedProgressOperation operation = new TimedProgressOperation("migrated memberships for group", total, log);

        TimedOperation getMembershipsOperation = new TimedOperation();
        final Collection<G> values = groupsByName.values();
        log.debug(getMembershipsOperation.complete("Got remote memberships"));
        Iterator<Membership> iter = getMemberships(values, isFullSync).iterator();
        TimedOperation applyMembershipsOperation = new TimedOperation();
        while (iter.hasNext()) {
            long start = System.currentTimeMillis();
            Membership membership = iter.next();
            long finish = System.currentTimeMillis();

            long duration = finish - start;

            log.debug("found [ " + membership.getUserNames().size() + " ] remote user-group memberships, "
                    + "[ " + membership.getChildGroupNames().size() + " ] remote group-group memberships in [ "
                    + duration + "ms ]");

            Group g = groupsByName.get(membership.getGroupName());
            if (g == null) {
                log.debug("Unexpected group in response: " + membership.getGroupName());
                continue;
            }

            directoryCache.syncUserMembersForGroup(g, membership.getUserNames());

            if (remoteDirectory.supportsNestedGroups()) {
                directoryCache.syncGroupMembersForGroup(g, membership.getChildGroupNames());
            }

            operation.incrementedProgress();
        }
        log.debug(applyMembershipsOperation.complete("Applied remote memberships"));
    }

    protected boolean isIncrementalSyncEnabled() {
        return Boolean.parseBoolean(remoteDirectory.getValue(SynchronisableDirectoryProperties.INCREMENTAL_SYNC_ENABLED));
    }

    protected String directoryDescription() {
        return remoteDirectory.getDescriptiveName() + " Directory " + remoteDirectory.getDirectoryId();
    }
}
