package com.atlassian.crowd.search.util;

import com.atlassian.crowd.search.query.entity.EntityQuery;

import java.util.List;
import java.util.function.Predicate;

/**
 * An aggregator across results from multiple queries that may include duplicates. A provided
 * key-making function provides a distinct sortable identifier for each result, allowing duplicates
 * to be excluded and results to be provided in the correct order.
 *
 * @param <T> the type of the individual results
 */
public interface ResultsAggregator<T> {
    /**
     * Include the given result in the total, with earlier results taking precedence. This means
     * that any later result that uses the same key as an earlier one will be ignored.
     *
     * @param result a single result
     */
    void add(T result);

    /**
     * Include all the given results in the total, with earlier results taking precedence. This means
     * that any later result that uses the same key as an earlier one will be ignored.
     *
     * @param results an iterable collection of results
     */
    void addAll(Iterable<? extends T> results);

    /**
     * @return the current number of distinct results
     */
    int size();

    /**
     * Return the subset of results required by the initial query's paging parameters.
     *
     * @return a subset of the results
     */
    List<T> constrainResults();

    /**
     * Return the subset of results required by the initial query's paging parameters which also satisfy the provided
     * criteria.
     *
     * @param criteria to be used when creating the subset of results to be returned.
     * @return a subset of the results
     */
    List<T> constrainResults(Predicate<? super T> criteria);


    /**
     * Return the total number of results we need to fetch for this query. This will be <code>start + maxResults</code>
     * from the original query, or {@link EntityQuery#ALL_RESULTS} if that was the original maximum. This number
     * must be fetched from each underlying directory for correct aggregation.
     *
     * @return how many results need to be fetched overall
     */
    int getRequiredResultCount();
}
