package com.atlassian.crowd.search.util;

import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.search.query.entity.EntityQuery;

import java.util.List;
import java.util.Optional;

/**
 * Utility class for executing split queries and merging results.
 */
public class QuerySplitter {
    /**
     * Splits the query if needed, executes it, aggregates and returns results. Query is split if the number
     * of OR conditions is higher than {@code maxRestrictionsPerQuery}. Only top level condition split is supported.
     * @param query original query to be executed
     * @param searcher query results provider
     * @param maxRestrictionsPerQuery maximum number of restrictions allowed in the single query
     * @param <T> result type
     * @return results of the original {@code query}
     * @throws OperationFailedException rethrows exception from {@code searcher}
     */
    public static <T, E extends Exception> List<T> batchConditionsIfNeeded(
            EntityQuery<T> query, Searcher<T, E> searcher, int maxRestrictionsPerQuery) throws E {
        Optional<List<EntityQuery<T>>> split = query.splitOrRestrictionIfNeeded(maxRestrictionsPerQuery);
        if (split.isPresent()) {
            ResultsAggregator<T> aggregator = ResultsAggregators.with(query);
            for (EntityQuery<T> singleQuery : split.get()) {
                aggregator.addAll(searcher.search(singleQuery));
            }
            return aggregator.constrainResults();
        } else {
            return searcher.search(query);
        }
    }

    public interface Searcher<T, E extends Exception> {
        List<T> search(EntityQuery<T> query) throws E;
    }
}
