package com.atlassian.crowd.util;

import java.net.InetAddress;

import javax.annotation.Nullable;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.atlassian.cache.Cache;
import com.atlassian.crowd.service.cluster.ClusterMessageListener;
import com.atlassian.crowd.service.cluster.ClusterMessageService;
import com.atlassian.crowd.manager.cluster.message.SingleClusterMessageListener;
import com.atlassian.crowd.model.application.Application;

/**
 * Utility class to store in a cache whether the InetAddress is permitted or forbidden to make a request to the Crowd server.
 */
public class ClusterAwareInetAddressCache {
    protected static final String CLUSTER_MESSAGE_CHANNEL = "ClusterAwareInetAddressCache";
    protected static final String CLUSTER_MESSAGE_CLEAR = "clear";

    private final Cache<String, Boolean> cache;
    private final ClusterMessageService clusterMessageService;
    private final ClusterMessageListener clusterMessageListener =
            new SingleClusterMessageListener(CLUSTER_MESSAGE_CHANNEL, CLUSTER_MESSAGE_CLEAR, this::clearLocal);

    public ClusterAwareInetAddressCache(Cache<String, Boolean> cache, ClusterMessageService clusterMessageService) {
        this.cache = cache;
        this.clusterMessageService = clusterMessageService;
    }

    @PostConstruct
    public void registerClusterListener() {
        clusterMessageService.registerListener(clusterMessageListener, CLUSTER_MESSAGE_CHANNEL);

    }

    @PreDestroy
    public void unregisterClusterListener() {
        clusterMessageService.unregisterListener(clusterMessageListener);
    }

    /**
     * Sets in the cache that <code>address</code> is permitted/forbidden from making a request to the Crowd server.
     *
     * @param application application that is requesting the connection.
     * @param address     InetAddress to set
     * @param permitted   whether <code>address</code> is permitted or forbidden to make a request.
     */
    public void setPermitted(Application application, InetAddress address, boolean permitted) {
        cache.put(getKeyName(application, address), permitted);
    }

    /**
     * Gets from cache whether the <code>application</code> with <code>address</code> is permitted to make a request
     * to the Crowd server.
     *
     * @param application Application making the request.
     * @param address     address of the client making the request.
     * @return an indication of whether the <code>application</code> with <code>address</code> is permitted to make a request, or <code>null</code> if unknown
     */
    @Nullable
    public Boolean getPermitted(Application application, InetAddress address) {
        return cache.get(getKeyName(application, address));
    }

    /**
     * Clears the entire cache storing the address permissions.
     */
    public void clear() {
        clusterMessageService.publish(CLUSTER_MESSAGE_CHANNEL, CLUSTER_MESSAGE_CLEAR);
        clearLocal();
    }

    private void clearLocal() {
        cache.removeAll();
    }

    /**
     * Returns the name of the cache key, given the application and the address.
     *
     * @param application Application
     * @param address     InetAddress
     * @return name of the key
     */
    private static String getKeyName(Application application, InetAddress address) {
        return application.getName() + "#" + address.getHostAddress();
    }
}
