package com.atlassian.crowd.search.util;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.search.query.entity.EntityQuery;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * An aggregator across results from multiple queries that may include duplicates. A provided
 * key-making function provides a distinct sortable identifier for each result, allowing duplicates
 * to be excluded and results to be provided in the correct order.
 *
 * @param <K> Comparable key type. Main purpose of this parameter is to make it invisible from caller's perspective.
 * @param <T> The type of the individual results.
 */

/**
 * An implementation of {@link ResultsAggregator}, kept as a separate class to hide the generic
 * key parameter from users.
 */
class ResultsAggregatorImpl<T, K extends Comparable<? super K>> implements ResultsAggregator<T> {
    private final int startIndex, maxResults;
    private final int totalResults;
    private final Function<? super T, ? extends K> keymaker;
    private final Map<K, T> contents;

    ResultsAggregatorImpl(Function<? super T, ? extends K> keymaker, Query<? extends T> query) {
        this(keymaker, query.getStartIndex(), query.getMaxResults());
    }

    ResultsAggregatorImpl(Function<? super T, ? extends K> keymaker, int startIndex, int maxResults) {
        this.startIndex = startIndex;
        this.maxResults = maxResults;
        this.totalResults = EntityQuery.addToMaxResults(maxResults, startIndex);
        this.keymaker = keymaker;
        this.contents = new HashMap<>();
    }

    @Override
    public void add(T t) {
        K k = keymaker.apply(t);
        contents.putIfAbsent(k, t);
    }

    @Override
    public void addAll(Iterable<? extends T> results) {
        for (T t : results) {
            add(t);
        }
    }

    @Override
    public List<T> constrainResults() {
        return constrainResults(t -> true);
    }

    @Override
    public List<T> constrainResults(Predicate<? super T> criteria) {
        return contents.entrySet().stream()
                .sorted(Map.Entry.comparingByKey())
                .map(Map.Entry::getValue)
                .filter(criteria)
                .skip(startIndex)
                .limit(EntityQuery.allResultsToLongMax(maxResults))
                .collect(Collectors.toList());
    }

    @Override
    public int size() {
        return contents.size();
    }

    @Override
    public int getRequiredResultCount() {
        return totalResults;
    }
}
