package com.atlassian.crowd.manager.login;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.ApplicationPermissionException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidEmailAddressException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryPermissionException;
import com.atlassian.crowd.manager.login.exception.InvalidResetPasswordTokenException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.token.ExpirableUserToken;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Manages functionality related to retrieving forgotten usernames or resetting forgotten passwords.
 *
 * <p>To reset a user's password, clients of {@link ForgottenLoginManager} would do the following:
 * <ol>
 * <li><tt>sendResetLink</tt> sends the user a unique link to reset their password</li>
 * <li><tt>resetUserCredential</tt> verifies that the reset token given by the user is correct using
 * <tt>isValidResetToken</tt>, then resets if the user credentials if the token is valid.</li>
 * </ol>
 *
 * @since v2.1.0
 */
public interface ForgottenLoginManager {
    // The default expiry time should be long enough to be usable, but as short as possible so you don't have
    // valid tokens hanging around as possible attack vectors. As a 'reset password' token is for immediate
    // consumption, 1 day seems reasonable.
    int DEFAULT_TOKEN_EXPIRY_SECONDS = (int) TimeUnit.HOURS.toSeconds(24);

    /**
     * Sends a reset link to the first user with the matching <tt>username</tt> from all the active directories assigned
     * to the application.
     *
     * @param application        user is searched in <tt>application</tt>'s assigned directories
     * @param username           username of the user to send the password reset link
     * @param tokenExpirySeconds number of seconds before generated token expires, or DEFAULT_TOKEN_EXPIRY_SECONDS
     * @throws UserNotFoundException              if no user with the supplied username exists
     * @throws InvalidEmailAddressException       if the user does not have a valid email address to send the password reset email to
     * @throws ApplicationPermissionException     if the application does not have permission to modify the user
     * @throws java.lang.IllegalArgumentException if tokenExpirySeconds is less than 0
     */
    void sendResetLink(Application application, String username, int tokenExpirySeconds)
            throws UserNotFoundException, InvalidEmailAddressException, ApplicationPermissionException;

    /**
     * <p>Sends the usernames associated with the given email address. No email will be sent if there are no usernames
     * associated with a given <code>email</code>.</p>
     * <p>The method returns a boolean, which should only ever be passed to authenticated applications to avoid
     * leaking information.</p>
     *
     * @param application search application's assigned directories for usernames associated with the <code>email</code>
     * @param email       email address of the user
     * @return <code>true</code> if any users with that address were found.
     * @throws InvalidEmailAddressException if the <code>email</code> is not valid
     */
    boolean sendUsernames(Application application, String email) throws InvalidEmailAddressException;

    /**
     * Sends a reset link to the user with specified username and directory ID.
     *
     * <p>Similar to {@link ForgottenLoginManager#sendResetLink(Application, String, int)} except applying to a directory-specific
     * user.
     *
     * @param directoryId        directory ID of the user to modify
     * @param username           username of the user to send the password reset link
     * @param tokenExpirySeconds number of seconds before generated token expires, or DEFAULT_TOKEN_EXPIRY_SECONDS
     * @throws DirectoryNotFoundException         if the directory specified by <tt>directoryId</tt> could not be found
     * @throws UserNotFoundException              if the user specified by <tt>username</tt> could not be found
     * @throws InvalidEmailAddressException       if the user does not have a valid email address to send the password reset email to
     * @throws java.lang.IllegalArgumentException if tokenExpirySeconds is less than 0
     */
    void sendResetLink(long directoryId, String username, int tokenExpirySeconds)
            throws DirectoryNotFoundException, UserNotFoundException, InvalidEmailAddressException, OperationFailedException;

    /**
     * Returns <tt>true</tt> if the password reset token for the user with the specified username and directory ID are
     * valid and not expired.  The valid password reset token is created by {@link ForgottenLoginManager#sendResetLink}.
     *
     * @param directoryId directory ID of the user to validate
     * @param username    username of the user to verify the <tt>token</tt>
     * @param token       password reset token
     * @return <tt>true</tt> if the username and reset token are a valid combination and the reset token has not expired.
     */
    boolean isValidResetToken(long directoryId, String username, String token);

    /**
     * Resets the user credentials and invalidates the token.
     *
     * @param directoryId directory ID of the user
     * @param username    user name of the user to perform a credential reset
     * @param credential  new credentials
     * @param token       password reset token
     * @throws DirectoryNotFoundException         if the directory could not be found.
     * @throws UserNotFoundException              if the user could not be found in the given directory.
     * @throws InvalidResetPasswordTokenException if the reset token is not valid.
     * @throws OperationFailedException           if there was an error performing the operation or instantiating the backend directory.
     * @throws InvalidCredentialException         if the user's credential does not meet the validation requirements for an associated directory.
     * @throws DirectoryPermissionException       if the directory is not allowed to perform the operation
     */
    void resetUserCredential(long directoryId, String username, PasswordCredential credential, String token)
            throws DirectoryNotFoundException, UserNotFoundException, InvalidResetPasswordTokenException, OperationFailedException, InvalidCredentialException, DirectoryPermissionException;

    /**
     * Creates an {@link ExpirableUserToken} for a given username in the given directory.
     * Note: no check is done to verify that the user actually exists in the given directory;
     * if this is not the case, the returned token will be useless.
     *
     * @param directoryId        the directory id associated with the user
     * @param username           the username of the user to create the token for
     * @param email              the email of the user to create the token for
     * @param tokenExpirySeconds number of seconds before generated token expires, or DEFAULT_TOKEN_EXPIRY_SECONDS
     * @return The ExpirableUserToken
     * @throws java.lang.IllegalArgumentException if tokenExpirySeconds is less than 0
     */
    ExpirableUserToken createAndStoreResetToken(long directoryId, String username, String email, int tokenExpirySeconds);

    /**
     * Removes the password reset tokens associated to a username in a directory.
     *
     * @param directoryId directory where the user lives
     * @param username    username
     * @return <tt>true</tt> if some tokens were removed
     */
    boolean removeByDirectoryAndUsername(long directoryId, String username);

    /**
     * Check if user is active
     *
     * @param directoryId directory where the user lives
     * @param username    username
     * @return <tt>true</tt> if user is active
     */
    boolean isUserActive(long directoryId, String username);

    /**
     * Returns password reset token for given user. For testing purposes only.
     */
    @ExperimentalApi
    Optional<ExpirableUserToken> getToken(long directoryId, String username);
}
