package com.atlassian.crowd.util;

import com.atlassian.cache.Cache;
import com.atlassian.crowd.model.application.Application;

import java.net.InetAddress;

import javax.annotation.Nullable;

/**
 * Utility class to store in a cache whether the InetAddress is permitted or forbidden to make a request to the Crowd server.
 */
public class InetAddressCacheUtil {
    public static final String INET_ADDRESS_CACHE_NAME = InetAddressCacheUtil.class.getName() + "_cache";

    private final Cache<String, Boolean> cache;

    public InetAddressCacheUtil(Cache<String, Boolean> cache) {
        this.cache = cache;
    }

    /**
     * Sets in the cache that <code>address</code> is permitted/forbidden from making a request to the Crowd server.
     *
     * @param application application that is requesting the connection.
     * @param address     InetAddress to set
     * @param permitted   whether <code>address</code> is permitted or forbidden to make a request.
     */
    public void setPermitted(Application application, InetAddress address, boolean permitted) {
        cache.put(getKeyName(application, address), permitted);
    }

    /**
     * Gets from cache whether the <code>application</code> with <code>address</code> is permitted to make a request
     * to the Crowd server.
     *
     * @param application Application making the request.
     * @param address     address of the client making the request.
     * @return an indication of whether the <code>application</code> with <code>address</code> is permitted to make a request, or <code>null</code> if unknown
     */
    @Nullable
    public Boolean getPermitted(Application application, InetAddress address) {
        return cache.get(getKeyName(application, address));
    }

    /**
     * Clears the entire cache storing the address permissions.
     */
    public void clearCache() {
        cache.removeAll();
    }

    /**
     * Returns the name of the cache key, given the application and the address.
     *
     * @param application Application
     * @param address     InetAddress
     * @return name of the key
     */
    private static String getKeyName(Application application, InetAddress address) {
        return application.getName() + "#" + address.getHostAddress();
    }
}
