package com.atlassian.crowd.manager.validation;

import java.net.InetAddress;

import com.atlassian.crowd.manager.property.PropertyManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.util.InetAddressCacheUtil;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Caches the result of validating the application remote address. Caching is enabled or disabled
 * depending on {@link PropertyManager#isCacheEnabled()}.
 */
public class CachedApplicationRemoteAddressValidator implements ApplicationRemoteAddressValidator {
    private final PropertyManager propertyManager;
    private final InetAddressCacheUtil cacheUtil;
    private final ApplicationRemoteAddressValidator delegate;

    public CachedApplicationRemoteAddressValidator(PropertyManager propertyManager,
                                                   InetAddressCacheUtil cacheUtil,
                                                   ApplicationRemoteAddressValidator delegate) {
        this.delegate = checkNotNull(delegate);
        this.propertyManager = checkNotNull(propertyManager);
        this.cacheUtil = checkNotNull(cacheUtil);
    }

    @Override
    public boolean validate(Application application, InetAddress clientAddress) {
        if (propertyManager.isCacheEnabled()) {
            Boolean addressValid = cacheUtil.getPermitted(application, clientAddress);
            if (addressValid == null) {
                addressValid = delegate.validate(application, clientAddress);
                cacheUtil.setPermitted(application, clientAddress, addressValid);
            }
            return addressValid;
        } else {
            return delegate.validate(application, clientAddress);
        }
    }
}
