package com.atlassian.crowd.service;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.FeatureInaccessibleException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.query.entity.EntityQuery;

import java.util.List;

/**
 * Interface for various user searches.
 * @since 3.3.0
 */
@ExperimentalApi
public interface DirectoryEntitySearchService {
    /**
     * Returns users matching {@code usersQuery}.
     * Requires a user context, and only allows searches in directories that the current user can administer.
     *
     * @param usersQuery users query
     * @param directoriesRestriction if not empty, search will be restricted to the specified directories
     * @return list of matching users with stable sort. User existing in multiple directories won't be merged.
     * @throws DirectoryNotFoundException
     * @throws OperationFailedException
     * @throws IllegalAccessException
     */
    List<User> searchUsers(EntityQuery<User> usersQuery, List<Directory> directoriesRestriction)
            throws DirectoryNotFoundException, OperationFailedException, IllegalAccessException, UserNotFoundException,
                   FeatureInaccessibleException;

    /**
     * Returns users that can be assigned as members of the specified group.
     * @param usersQuery users query
     * @param directoryId directory id of the group
     * @param groupName group name
     * @return list of matching users with stable sort.
     * @throws DirectoryNotFoundException
     * @throws GroupNotFoundException
     * @throws OperationFailedException
     * @throws UserNotFoundException
     */
    List<User> searchPotentialMembers(EntityQuery<User> usersQuery, long directoryId, String groupName)
            throws DirectoryNotFoundException, GroupNotFoundException, OperationFailedException, UserNotFoundException,
                   FeatureInaccessibleException;

    /**
     * Returns groups matching {@code groupsQuery}.
     * Requires a user context, and only allows searches in directories that the current user can administer.
     *
     * @param groupsQuery groups query
     * @param directoriesRestriction if not empty, search will be restricted to the specified directories
     * @return list of matching groups with stable sort. Group existing in multiple directories won't be merged.
     * @throws DirectoryNotFoundException
     * @throws OperationFailedException
     * @throws IllegalAccessException
     */
    List<Group> searchGroups(EntityQuery<Group> groupsQuery, List<Directory> directoriesRestriction)
            throws DirectoryNotFoundException, OperationFailedException, IllegalAccessException, UserNotFoundException,
                   FeatureInaccessibleException;
}
