package com.atlassian.crowd.service;

import com.atlassian.crowd.manager.rememberme.CrowdSpecificRememberMeSettings;
import org.springframework.security.core.Authentication;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Optional;

/**
 * Service allowing storage, retrieval and manipulation of {@link com.atlassian.crowd.model.rememberme.CrowdRememberMeToken}
 *
 * @since v3.4
 */
public interface CrowdRememberMeService {

    /**
     * Attempts to authenticate a request using the remember-me cookie provided on the request. If authentication
     * succeeds, a new remember-me token is created for the user and a new cookie is set on the response. If
     * authentication fails for any reason, the cookie is cleared on the response.
     *
     * @param request  the request
     * @param response the response
     * @return the authenticated user, or {@code Optional.empty()} if authentication failed.
     */
    Optional<Authentication> authenticate(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response);

    /**
     * Clears the remember-me cookie and removes all remember-me tokens for the corresponding series
     *
     * @param request  the request
     * @param response the response
     */
    void removeCookie(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response);

    /**
     * Generates a new remember-me token for the user and sets the corresponding cookie on the response. This will
     * cancel an existing remember me cookie if such was provided..
     *
     * @param authentication valid authentication containing the user context
     * @param request        the request
     * @param response       the response
     */
    void createCookie(@Nonnull Authentication authentication, @Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response);

    /**
     * Clears all remember me tokens for the user with given series
     */
    void clearAllTokensForSeries(String series);

    /**
     * Clears all remember me tokens for the user with given username, and directory id
     */
    void clearAllTokensForUserInDirectory(String username, Long directoryId);

    /**
     * Clears all remember me tokens for all the users
     */
    void clearAllTokens();

    /**
     * Clears all expired remember me tokens
     */
    void clearAllExpiredTokens();

    /**
     * Clears all remember me tokens for the users belonging to the given directoryId
     */
    void clearAllTokensForDirectory(Long directoryId);

    /**
     * Validates and stores passed configuration, if {@link CrowdSpecificRememberMeSettings#isEnabled()} happens to be
     * {@code false} all existing tokens will be expired immediately
     *
     * @param configuration new configuration
     */
    void saveConfiguration(CrowdSpecificRememberMeSettings configuration);

    /**
     * Retrieves current configuration
     *
     * @return configuration
     */
    CrowdSpecificRememberMeSettings getConfiguration();
}
