package com.atlassian.crowd.manager.mail;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import javax.activation.DataSource;
import javax.mail.internet.InternetAddress;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * Represents a plain text email message.
 */
public class TextEmailMessage implements EmailMessage {
    private final InternetAddress from;
    private final Collection<InternetAddress> to;
    private final Collection<InternetAddress> cc;
    private final Collection<InternetAddress> bcc;
    private final Collection<InternetAddress> replyTo;
    private final String body;
    private final String subject;
    private final Map<String, DataSource> attachments;
    private final Map<String, String> headers;

    protected TextEmailMessage(Builder builder) {
        this.from = builder.from;
        this.to = builder.to != null ? builder.to : Collections.emptyList();
        this.cc = builder.cc != null ? builder.cc : Collections.emptyList();
        this.bcc = builder.bcc != null ? builder.bcc : Collections.emptyList();
        this.replyTo = builder.replyTo != null ? builder.replyTo : Collections.emptyList();
        this.body = builder.body != null ? builder.body : "";
        this.subject = builder.subject != null ? builder.subject : "";
        this.attachments = builder.attachments != null ? builder.attachments : Collections.emptyMap();
        this.headers = builder.headers != null ? builder.headers : Collections.emptyMap();

        Preconditions.checkArgument(!to.isEmpty() || !cc.isEmpty() || !bcc.isEmpty(), "No recipients chosen for the e-mail");
    }

    @Override
    public Optional<InternetAddress> getFrom() {
        return Optional.ofNullable(from);
    }

    @Override
    public Collection<InternetAddress> getTo() {
        return to;
    }

    @Override
    public Collection<InternetAddress> getCc() {
        return cc;
    }

    @Override
    public Collection<InternetAddress> getBcc() {
        return bcc;
    }

    @Override
    public Collection<InternetAddress> getReplyTo() {
        return replyTo;
    }

    @Override
    public String getBody() {
        return body;
    }

    @Override
    public String getSubject() {
        return subject;
    }

    @Override
    public Map<String, String> getHeaders() {
        return headers;
    }

    @Override
    public Map<String, DataSource> getAttachments() {
        return attachments;
    }

    public static TextEmailMessage.Builder builder() {
        return new TextEmailMessage.Builder();
    }

    public static TextEmailMessage.Builder builder(TextEmailMessage textEmailMessage) {
        return new TextEmailMessage.Builder(textEmailMessage);
    }

    public static class Builder {
        protected InternetAddress from;
        protected Collection<InternetAddress> to;
        protected Collection<InternetAddress> cc;
        protected Collection<InternetAddress> bcc;
        protected Collection<InternetAddress> replyTo;
        protected String body;
        protected String subject;
        protected Map<String, DataSource> attachments;
        protected Map<String, String> headers;

        protected Builder() {
        }

        protected Builder(TextEmailMessage textEmailMessage) {
            this.from = textEmailMessage.from;
            this.to = textEmailMessage.to;
            this.cc = textEmailMessage.cc;
            this.bcc = textEmailMessage.bcc;
            this.replyTo = textEmailMessage.replyTo;
            this.body = textEmailMessage.body;
            this.subject = textEmailMessage.subject;
            this.attachments = textEmailMessage.attachments;
            this.headers = textEmailMessage.headers;
        }

        public Builder setFrom(InternetAddress from) {
            this.from = from;
            return this;
        }

        /**
         * @deprecated since 3.6 use {@link #setTo}
         */
        @Deprecated
        public Builder setRecipientAddress(InternetAddress recipientAddress) {
            return setTo(Collections.singletonList(recipientAddress));
        }

        public Builder setTo(Iterable<InternetAddress> to) {
            this.to = Lists.newArrayList(to);
            return this;
        }

        public Builder setCc(Iterable<InternetAddress> cc) {
            this.cc = Lists.newArrayList(cc);
            return this;
        }

        public Builder setBcc(Iterable<InternetAddress> bcc) {
            this.bcc = Lists.newArrayList(bcc);
            return this;
        }

        public Builder setReplyTo(Iterable<InternetAddress> replyTo) {
            this.replyTo = Lists.newArrayList(replyTo);
            return this;
        }

        public Builder setBody(String body) {
            this.body = body;
            return this;
        }

        public Builder setSubject(String subject) {
            this.subject = subject;
            return this;
        }

        public Builder setAttachments(Map<String, DataSource> attachments) {
            this.attachments = attachments;
            return this;
        }

        public Builder setHeaders(Map<String, String> headers) {
            this.headers = headers;
            return this;
        }

        public TextEmailMessage build() {
            return new TextEmailMessage(this);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof TextEmailMessage)) {
            return false;
        }
        final TextEmailMessage that = (TextEmailMessage) o;
        return Objects.equals(from, that.from) &&
                Objects.equals(to, that.to) &&
                Objects.equals(cc, that.cc) &&
                Objects.equals(bcc, that.bcc) &&
                Objects.equals(replyTo, that.replyTo) &&
                Objects.equals(body, that.body) &&
                Objects.equals(subject, that.subject) &&
                Objects.equals(attachments, that.attachments) &&
                Objects.equals(headers, that.headers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(from, to, cc, bcc, replyTo, body, subject, attachments, headers);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE)
                .append("from", from)
                .append("to", to)
                .append("cc", cc)
                .append("bcc", bcc)
                .append("replyTo", replyTo)
                .append("body", body)
                .append("subject", subject)
                .append("attachments", attachments)
                .append("headers", headers)
                .toString();
    }
}