package com.atlassian.crowd.manager.permission;


import java.util.Collection;
import java.util.Set;

import javax.annotation.Nullable;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.permission.UserPermission;
import com.atlassian.crowd.model.user.User;

/**
 * Check Crowd access permissions for users, eg SYS_ADMIN and ADMIN.
 */
public interface UserPermissionService {
    /**
     * Does the current user have the provided permission.
     *
     * @param permission permission required
     * @return true if the user has permission
     */
    boolean currentUserHasPermission(UserPermission permission);

    /**
     * Does the provided user have the provided permission.
     *
     * @param username   user to check, null for the anonymous user
     * @param permission permission required
     * @return true if the user has permission
     */
    boolean hasPermission(@Nullable String username, UserPermission permission);

    /**
     * Does the provided user have the provided permission when not considering the provided groups.
     * <p>
     * This method is intended to be used to determine if a group is exclusively providing a given permission.
     *
     * @param username       user to check, null for the anonymous user
     * @param permission     permission required
     * @param excludedGroups groups to exclude in permission check, or an empty list otherwise
     * @return true if the user has permission outside of the provided groups
     */
    boolean hasPermissionOutsideOfGroups(@Nullable String username, UserPermission permission, Collection<DirectoryGroup> excludedGroups);

    /**
     * Does the provided user have Group Level Admin permissions.
     *
     * @param username user to check, null for anonymous user
     * @return true if the user has Group Level Admin permissions
     */
    boolean isGroupLevelAdmin(@Nullable String username);

    /**
     * @param application application the permission applies to
     * @param includeLocallyCachedOnly if set to true will limit search to users stored locally
     * @return users with SYS_ADMIN permission.
     */
    @ExperimentalApi
    default Set<User> getSysAdmins(Application application, boolean includeLocallyCachedOnly)
            throws DirectoryNotFoundException, OperationFailedException {
        return getUsersWithPermission(application, UserPermission.SYS_ADMIN, includeLocallyCachedOnly);
    }

    /**
     * @param application application the permission applies to
     * @param userPermission permission to check
     * @param includeLocallyCachedOnly if set to true will limit search to users stored locally
     * @return users with the given permission
     * @since 3.7
     */
    @ExperimentalApi
    Set<User> getUsersWithPermission(Application application, UserPermission userPermission, boolean includeLocallyCachedOnly)
            throws DirectoryNotFoundException, OperationFailedException;

    /**
     * @param considerOnlyTargetGroupsWithApplications whether groups without any active application should be taken into account
     * @return users with group level admin permission.
     */
    @ExperimentalApi
    Set<User> getGroupLevelAdmins(boolean considerOnlyTargetGroupsWithApplications)
            throws DirectoryNotFoundException, OperationFailedException;
}