package com.atlassian.crowd.plugin.rest.util;

import com.atlassian.crowd.embedded.api.Attributes;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.application.ApplicationService;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.plugin.rest.entity.UserEntity;
import com.atlassian.plugins.rest.common.Link;

import com.google.common.base.Preconditions;

import org.apache.commons.lang3.Validate;

/**
 * Utility class for UserEntity.
 */
public class UserEntityUtil {
    private UserEntityUtil() {
        // prevent instantiation
    }

    /**
     * Expands a UserEntity from its minimal form to the expanded version. Attributes are expanded if
     * <tt>expandAttributes</tt> is <tt>true</tt>, otherwise, UserEntity is returned with no attributes expanded.
     * <p>
     * N.B. This is not the same as expanding a user entity, which expands a UserEntity from its minimal form, to having
     * all the first name, last name, email, etc filled in. Expanding a UserEntity is automatically performed in {@link
     * com.atlassian.crowd.plugin.rest.entity.UserEntityExpander}.
     *
     * @param applicationService ApplicationService to find a user
     * @param application        name of the application requesting the user
     * @param minimalUserEntity  Minimal representation of a UserEntity. Must include at least a username and a link.
     * @param expandAttributes   set to true if the expanded UserEntity should expand attributes.
     * @return UserEntity expanded UserEntity
     * @throws IllegalArgumentException if the minimal UserEntity does not include at least a username and a link.
     * @throws UserNotFoundException    if a user with the specified username could not be found.
     */
    public static UserEntity expandUser(final ApplicationService applicationService, final Application application, final UserEntity minimalUserEntity, final boolean expandAttributes)
            throws UserNotFoundException {
        Validate.notNull(applicationService);
        Validate.notNull(application);
        Validate.notNull(minimalUserEntity);
        Validate.notNull(minimalUserEntity.getName(), "Minimal user entity must include a username");
        Validate.notNull(minimalUserEntity.getLink(), "Minimal user entity must include a link");

        final String username = minimalUserEntity.getName();
        final Link userLink = minimalUserEntity.getLink();

        UserEntity expandedUser;
        if (expandAttributes) {
            UserWithAttributes user = applicationService.findUserWithAttributesByName(application, username);
            Link updatedLink = LinkUriHelper.updateUserLink(userLink, user.getName()); // use the canonical username in the link
            expandedUser = EntityTranslator.toUserEntity(user, user, updatedLink);
        } else {
            User user = applicationService.findUserByName(application, username);
            Link updatedLink = LinkUriHelper.updateUserLink(userLink, user.getName()); // use the canonical username in the link
            expandedUser = EntityTranslator.toUserEntity(user, updatedLink);
        }
        return expandedUser;
    }

    public static UserEntity translate(User user, Link oldLink) {
        Link updatedLink = LinkUriHelper.updateUserLink(oldLink, user.getName()); // use the canonical username in the link
        return EntityTranslator.toUserEntity(user, updatedLink);
    }

    public static UserEntity translateWithAttributes(User user, Attributes attributes, Link oldLink) {
        Link updatedLink = LinkUriHelper.updateUserLink(oldLink, user.getName()); // use the canonical username in the link
        return EntityTranslator.toUserEntity(user, attributes, updatedLink);
    }

    public static UserEntity expandUser(final DirectoryManager directoryManager, final long directoryId, final UserEntity minimalUserEntity, final boolean expandAttributes)
            throws UserNotFoundException, DirectoryNotFoundException, OperationFailedException {
        Preconditions.checkNotNull(directoryManager);
        Preconditions.checkNotNull(directoryManager);
        Preconditions.checkArgument(directoryId > 0, "The directory id must be greater than 0, %d", directoryId);
        Preconditions.checkNotNull(minimalUserEntity);
        Preconditions.checkNotNull(minimalUserEntity.getName(), "Minimal user entity must include a username");
        Preconditions.checkNotNull(minimalUserEntity.getLink(), "Minimal user entity must include a link");

        final String username = minimalUserEntity.getName();
        final Link userLink = minimalUserEntity.getLink();

        final UserEntity expandedUser;
        if (expandAttributes) {
            UserWithAttributes user = directoryManager.findUserWithAttributesByName(directoryId, username);
            Link updatedLink = LinkUriHelper.updateUserLink(userLink, user.getName()); // use the canonical username in the link
            expandedUser = EntityTranslator.toUserEntity(user, user, updatedLink);
        } else {
            User user = directoryManager.findUserByName(directoryId, username);
            Link updatedLink = LinkUriHelper.updateUserLink(userLink, user.getName()); // use the canonical username in the link
            expandedUser = EntityTranslator.toUserEntity(user, updatedLink);
        }
        expandedUser.setDirectoryId(directoryId);
        expandedUser.setDirectoryName(directoryManager.findDirectoryById(directoryId).getName());
        return expandedUser;
    }
}
