package com.atlassian.crowd.plugin.rest.entity.page;


import com.atlassian.crowd.model.page.Page;
import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.codehaus.jackson.annotate.JsonAutoDetect;
import org.codehaus.jackson.annotate.JsonProperty;

import java.util.List;
import java.util.Objects;

@JsonAutoDetect(getterVisibility = JsonAutoDetect.Visibility.NONE, isGetterVisibility = JsonAutoDetect.Visibility.NONE)
public class RestPage<T> implements Page<T> {
    public static final String LAST_PAGE_PROPERTY = "isLastPage";
    public static final String LIMIT_PROPERTY = "limit";
    public static final String START_PROPERTY = "start";
    public static final String SIZE_PROPERTY = "size";
    public static final String VALUES_PROPERTY = "values";

    @JsonProperty(VALUES_PROPERTY)
    private List<T> results;
    @JsonProperty(SIZE_PROPERTY)
    private int size;
    @JsonProperty(START_PROPERTY)
    private int start;
    @JsonProperty(LIMIT_PROPERTY)
    private int limit;
    @JsonProperty(LAST_PAGE_PROPERTY)
    private boolean isLastPage;

    protected RestPage() {
    }

    public <E> RestPage(Page<? extends E> page, Function<E, ? extends T> restTransform) {
        this(ImmutableList.copyOf(Iterables.transform(page.getResults(), restTransform)),
                page.getSize(), page.getStart(), page.getLimit(), page.isLastPage());
    }

    public RestPage(List<T> results, int size, int start, int limit, boolean isLastPage) {
        this.results = results;
        this.size = size;
        this.start = start;
        this.limit = limit;
        this.isLastPage = isLastPage;
    }

    public List<T> getResults() {
        return results;
    }

    public int getSize() {
        return size;
    }

    public int getStart() {
        return start;
    }

    public int getLimit() {
        return limit;
    }

    public boolean isLastPage() {
        return isLastPage;
    }

    public static <T> RestPage<T> fromListPlusOne(List<T> results, RestPageRequest pageRequest) {
        final int limit = pageRequest.getLimit();
        final int start = pageRequest.getStart();

        if (pageRequest.isAllResultsQuery()) {
            return new RestPage<>(results, results.size(), start, limit, true);
        } else {
            final List<T> trimmedResults = results.size() > limit ? results.subList(0, limit) : results;
            return new RestPage<T>(trimmedResults, trimmedResults.size(), start, limit, results.size() <= limit);
        }
    }

    public static <T, A> RestPage<T> fromListPlusOne(List<A> results, Function<A, ? extends T> transformer, RestPageRequest pageRequest) {
        return new RestPage<>(fromListPlusOne(results, pageRequest), transformer);
    }

    public static int limitPlusOne(int limit) {
        return (limit == RestPageRequest.ALL_RESULTS || limit == Integer.MAX_VALUE) ? limit : limit + 1;
    }

    /**
     * Converts max results limit to long value - replaces {@link #ALL_RESULTS} with {@link Long#MAX_VALUE}}.
     */
    public static long allResultsToLongMax(int maxResults) {
        return maxResults == RestPageRequest.ALL_RESULTS ? Long.MAX_VALUE : maxResults;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RestPage<?> restPage = (RestPage<?>) o;
        return size == restPage.size &&
                start == restPage.start &&
                limit == restPage.limit &&
                isLastPage == restPage.isLastPage &&
                Objects.equals(results, restPage.results);
    }

    @Override
    public int hashCode() {
        return Objects.hash(results, size, start, limit, isLastPage);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("results", results)
                .add("size", size)
                .add("start", start)
                .add("limit", limit)
                .add("isLastPage", isLastPage)
                .toString();
    }
}
