package com.atlassian.crowd.plugin.rest.entity.admin.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.plugin.rest.entity.admin.user.UserData;
import com.atlassian.crowd.plugin.rest.entity.directory.GroupAdministrationMappingRestDTO;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Helper class transforming directory entities between different formats.
 * Deals with {@link DirectoryNotFoundException}, which can't be handled by {@link java.util.stream.Stream#map(Function)}.
 */
public class DirectoryEntitiesTransformer {
    private final Function<Long, DirectoryData> directoryFinder;

    private DirectoryEntitiesTransformer(Function<Long, DirectoryData> directoryFinder) {
        this.directoryFinder = directoryFinder;
    }

    private <F, T> T transform(
            F entity,
            Function<F, Long> directoryIdGetter,
            BiFunction<F, DirectoryData, T> transformer) {
        DirectoryData directory = directoryFinder.apply(directoryIdGetter.apply(entity));
        return transformer.apply(entity, directory);
    }

    public List<UserData> toUserData(List<? extends User> users) {
        return users.stream().map(this::toUserData).collect(Collectors.toList());
    }

    public UserData toUserData(User user) {
        return transform(user, User::getDirectoryId, UserData::fromUser);
    }

    public UserData toUserDataWithAvatar(User user, String avatarUrl) {
        DirectoryData directoryData = directoryFinder.apply(user.getDirectoryId());
        return UserData.fromUserWithAvatarUrl(user, directoryData, avatarUrl);
    }

    public List<GroupAdministrationMappingRestDTO> fromUserToGroupAdministrationMappingRestDTO(List<? extends User> users) {
        return users.stream().map(this::toGroupAdministrationMappingRestDTO).collect(Collectors.toList());
    }

    public GroupAdministrationMappingRestDTO toGroupAdministrationMappingRestDTO(User user) {
        return transform(user, User::getDirectoryId, GroupAdministrationMappingRestDTO::fromUser);
    }

    public List<GroupAdministrationMappingRestDTO> fromGroupToGroupAdministrationMappingRestDTO(List<Group> users) {
        return users.stream().map(this::toGroupAdministrationMappingRestDTO).collect(Collectors.toList());
    }

    public GroupAdministrationMappingRestDTO toGroupAdministrationMappingRestDTO(Group group) {
        return transform(group, Group::getDirectoryId, GroupAdministrationMappingRestDTO::fromGroup);
    }

    public static DirectoryEntitiesTransformer withDirectoryCaching(DirectoryFinder directoryFinder) {
        Function<Long, DirectoryData> directoryDataFinder = toDirectoryDataFunction(directoryFinder);
        Map<Long, DirectoryData> directoryDataCache = new HashMap<>();
        return new DirectoryEntitiesTransformer(
                (Long id) -> directoryDataCache.computeIfAbsent(id, directoryDataFinder));
    }

    private static Function<Long, DirectoryData> toDirectoryDataFunction(DirectoryFinder finder) {
        return id -> {
            try {
                return DirectoryData.fromDirectory(finder.findById(id));
            } catch (DirectoryNotFoundException e) {
                throw new IllegalArgumentException();
            }
        };
    }

    @FunctionalInterface
    public interface DirectoryFinder {
        Directory findById(long id) throws DirectoryNotFoundException;
    }
}
