package com.atlassian.crowd.plugin.rest.exception.mapper;

import com.atlassian.crowd.plugin.rest.entity.ErrorEntity;
import com.google.common.collect.ImmutableMap;
import com.sun.jersey.core.util.ReaderWriter;
import org.apache.commons.lang3.ArrayUtils;

import javax.annotation.Nullable;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Optional;

@Provider
@Produces({"text/plain; charset=utf-8", "text/csv"})
public class ErrorEntityMessageBodyWriter implements MessageBodyWriter<ErrorEntity> {
    private static final MediaType textPlainUtf8Type = new MediaType("text", "plain", ImmutableMap.of("charset", "utf-8"));

    @Override
    public boolean isWriteable(Class<?> aClass, Type type, Annotation[] annotations, MediaType mediaType) {
        return type == ErrorEntity.class
                && (textPlainUtf8Type.equals(mediaType) || isTextCsv(mediaType))
                && aClass == ErrorEntity.class;
    }

    @Override
    public long getSize(@Nullable ErrorEntity errorEntity, Class<?> aClass, Type type, Annotation[] annotations, MediaType mediaType) {
        return Optional.ofNullable(errorEntity)
                .map(ErrorEntity::getMessage)
                .map(message -> toMessageBytes(message, mediaType))
                .map(bytes -> bytes.length)
                .orElse(0);
    }

    @Override
    public void writeTo(@Nullable ErrorEntity errorEntity, Class<?> aClass, Type type, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> multivaluedMap, OutputStream entityStream) throws IOException, WebApplicationException {
        if (errorEntity != null && errorEntity.getMessage() != null) {
            entityStream.write(toMessageBytes(errorEntity.getMessage(), mediaType));
        }
    }

    private byte[] toMessageBytes(final String message, final MediaType mediaType) {
        if (isTextCsv(mediaType)) {
            return ArrayUtils.EMPTY_BYTE_ARRAY; // No body for text/csv
        }
        return message.getBytes(ReaderWriter.getCharset(mediaType));
    }

    private boolean isTextCsv(final MediaType mediaType) {
        return mediaType != null
                && "text".equals(mediaType.getType())
                && "csv".equals(mediaType.getSubtype());
    }
}
