package com.atlassian.crowd.plugin.rest.entity.admin.directory;

import com.atlassian.crowd.common.util.Base64Util;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.group.Group;
import com.google.common.base.Splitter;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonValue;

import java.util.Base64;
import java.util.Iterator;
import java.util.Objects;

public class DirectoryEntityId {
    private static Splitter SPLITTER = Splitter.on("-").limit(2);

    private final long directoryId;
    private final String entityName;

    public DirectoryEntityId(long directoryId, String entityName) {
        this.directoryId = directoryId;
        this.entityName = entityName;
    }

    public long getDirectoryId() {
        return directoryId;
    }

    public String getEntityName() {
        return entityName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DirectoryEntityId that = (DirectoryEntityId) o;
        return directoryId == that.directoryId &&
                Objects.equals(entityName, that.entityName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(directoryId, entityName);
    }

    @JsonValue
    public String marshal() {
        return marshal(directoryId, entityName);
    }

    @Override
    public String toString() {
        return directoryId + "-" + entityName;
    }

    public static String marshal(long directoryId, String entityName) {
        final String entityNameEncoded = Base64Util.urlSafeEncoderWithoutPadding().encodeToString(entityName.getBytes());
        return directoryId + "-" + entityNameEncoded;
    }

    @JsonCreator
    public static DirectoryEntityId fromString(String identifier) {
        final Iterator<String> split = SPLITTER.split(identifier).iterator();
        final long directoryId = Long.parseLong(split.next());
        final String entityNameEncoded = split.next();
        final String entityName = new String(Base64.getUrlDecoder().decode(entityNameEncoded.getBytes()));
        return new DirectoryEntityId(directoryId, entityName);
    }

    public static DirectoryEntityId fromGroup(Group group) {
        return fromDirectoryEntity(group);
    }

    public static DirectoryEntityId fromUser(User user) {
        return new DirectoryEntityId(user.getDirectoryId(), user.getName());
    }

    public static DirectoryEntityId fromDirectoryEntity(DirectoryEntity entity) {
        return new DirectoryEntityId(entity.getDirectoryId(), entity.getName());
    }

    public static String marshal(DirectoryEntity entity) {
        return marshal(entity.getDirectoryId(), entity.getName());
    }
}
