package com.atlassian.crowd.plugin.rest.entity;

import com.atlassian.plugins.rest.common.Link;
import com.atlassian.plugins.rest.common.expand.Expandable;
import com.atlassian.plugins.rest.common.expand.Expander;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import java.util.Date;

/**
 * Represents a User entity (server side).
 *
 * @since v2.1
 */
@XmlRootElement(name = "user")
@XmlAccessorType(XmlAccessType.FIELD)
@Expander(UserEntityExpander.class)
public class UserEntity implements NamedEntity {
    /**
     * Name of the attributes field.
     */
    public static final String ATTRIBUTES_FIELD_NAME = "attributes";

    @SuppressWarnings("unused")
    @XmlAttribute
    private String expand;

    @XmlElement(name = "link")
    private Link link;

    @XmlAttribute(name = "name")
    private String name;

    @XmlElement(name = "first-name")
    private String firstName;

    @XmlElement(name = "last-name")
    private String lastName;

    @XmlElement(name = "display-name")
    private String displayName;

    /**
     * @since 2.8
     */
    @XmlElement(name = "directory-id")
    private Long directoryId;

    @XmlElement(name = "email")
    private String emailAddress;

    @XmlElement(name = "password")
    private PasswordEntity password; // will never be populated for a GET; only read for user creation or modification

    @XmlElement(name = "encrypted-password")
    private PasswordEntity encryptedPassword; // will never be populated for a GET; only read for user creation or modification

    @XmlElement(name = "key")
    private final String key;

    @XmlElement(name = "created-date")
    private Date createdDate;

    @XmlElement(name = "updated-date")
    private Date updatedDate;

    /**
     * Only used when creating a minimal UserEntity.
     *
     * @see {@link #newMinimalUserEntity(String, String, Link)}
     */
    @XmlTransient
    private String applicationName;

    /**
     * @since 2.8
     */
    @XmlElement(name = "directory-name")
    private String directoryName;

    @XmlElement(name = "active")
    private Boolean active;

    @Expandable
    @XmlElement(name = "attributes")
    private MultiValuedAttributeEntityList attributes;

    /**
     * JAXB requires a no-arg constructor.
     */
    private UserEntity() {
        this(null, null, null, null, null, null, null, null, null, null, null, null, null);
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final Boolean active, final Link link, final String key, final Long directoryId, final String directoryName, final Date createdDate, final Date updatedDate) {
        this(name, firstName, lastName, displayName, emailAddress, password, active, link, key, directoryId, directoryName, createdDate, updatedDate, false);
    }

    public UserEntity(final String name, final String firstName, final String lastName, final String displayName, final String emailAddress, final PasswordEntity password, final Boolean active, final Link link, final String key, final Long directoryId, final String directoryName, final Date createdDate, final Date updatedDate, boolean isPasswordEncrypted) {
        this.name = name;
        this.firstName = firstName;
        this.lastName = lastName;
        this.displayName = displayName;
        this.emailAddress = emailAddress;
        if (isPasswordEncrypted) {
            this.encryptedPassword = password;
            this.password = null;
        } else {
            this.password = password;
            this.encryptedPassword = null;
        }
        this.active = active;
        this.link = link;
        this.key = key;
        this.directoryId = directoryId;
        this.directoryName = directoryName;
        this.createdDate = createdDate;
        this.updatedDate = updatedDate;
    }

    public String getDisplayName() {
        return displayName;
    }

    public String getFirstName() {
        return firstName;
    }

    public String getLastName() {
        return lastName;
    }

    public void setEmail(final String email) {
        this.emailAddress = email;
    }

    public String getEmail() {
        return emailAddress;
    }

    public void setPassword(final PasswordEntity password) {
        this.password = password;
    }

    public PasswordEntity getPassword() {
        return password;
    }

    public void setEncryptedPassword(PasswordEntity encryptedPassword) {
        this.encryptedPassword = encryptedPassword;
    }

    public PasswordEntity getEncryptedPassword() {
        return encryptedPassword;
    }

    public void setActive(final boolean active) {
        this.active = active;
    }

    public Boolean isActive() {
        return active;
    }

    public void setName(final String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    public void setCreatedDate(Date createdDate) {
        this.createdDate = createdDate;
    }

    public Date getCreatedDate() {
        return createdDate;
    }

    public void setUpdatedDate(Date updatedDate) {
        this.updatedDate = updatedDate;
    }

    public Date getUpdatedDate() {
        return updatedDate;
    }

    public void setAttributes(final MultiValuedAttributeEntityList attributes) {
        this.attributes = attributes;
    }

    public MultiValuedAttributeEntityList getAttributes() {
        return attributes;
    }

    public String getKey() {
        return key;
    }

    /**
     * Returns the application name. Should only be used by
     * {@link com.atlassian.crowd.plugin.rest.entity.UserEntityExpander} to expand the UserEntity.
     *
     * @return application name
     */
    String getApplicationName() {
        return applicationName;
    }

    public String getDirectoryName() {
        return directoryName;
    }

    public void setDirectoryName(String directoryName) {
        this.directoryName = directoryName;
    }

    public void setDirectoryId(final Long directoryId) {
        this.directoryId = directoryId;
    }

    public Long getDirectoryId() {
        return directoryId;
    }

    /**
     * Creates a <tt>UserEntity</tt> with the minimal amount of information required.
     *
     * @param name            Username.
     * @param applicationName Name of the application.
     * @param link            Link to the canonical representation of the user. E.g. "/user?username=&lt;username&gt;".
     * @return UserEntity
     */
    public static UserEntity newMinimalUserEntity(final String name, final String applicationName, final Link link) {
        UserEntity user = new UserEntity(name, null, null, null, null, null, null, link, null, null, null, null, null);
        user.applicationName = applicationName;
        return user;
    }

    public String toString() {
        return new ToStringBuilder(this).
                append("name", getName()).
                append("active", isActive()).
                append("emailAddress", getEmail()).
                append("firstName", getFirstName()).
                append("lastName", getLastName()).
                append("displayName", getDisplayName()).
                append("key", getKey()).
                append("directoryId", getDirectoryId()).
                append("directoryName", getDirectoryName()).
                append("createdDate", getCreatedDate()).
                append("updatedDate", getUpdatedDate()).
                toString();
    }

    public void setLink(final Link link) {
        this.link = link;
    }

    public Link getLink() {
        return link;
    }

    /**
     * Does this object represent an expanded user, or does it only contain a username.
     *
     * @return true if this object represents an expanded user
     */
    public boolean isExpanded() {
        return applicationName == null;
    }
}
