package com.atlassian.crowd.dao.user;

import com.atlassian.crowd.embedded.spi.UserDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.user.InternalUser;
import com.atlassian.crowd.model.user.InternalUserWithPasswordLastChanged;
import com.atlassian.crowd.model.user.MinimalUser;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.util.persistence.hibernate.batch.BatchResultWithIdReferences;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Manages persistence of {@link User}.
 */
public interface InternalUserDao extends UserDao {
    BatchResultWithIdReferences<User> addAll(final Collection<UserTemplateWithCredentialAndAttributes> users);

    void removeAll(long directoryId) throws DirectoryNotFoundException;

    /**
     * Get all usernames for users belonging to a given directory
     *
     * @param directoryId
     * @return list of usernames
     */
    List<String> getAllUsernames(long directoryId);

    /**
     * Finds and return the user with given name and director ID.
     *
     * @throws UserNotFoundException if the user could not be found
     */
    InternalUser findByName(long directoryId, String userName) throws UserNotFoundException;

    /**
     * Bulk find of users using SQL disjunction. This will return fully populated InternalUser instances with names,
     * E-mail, Password credential records and so on. If those aren't needed then {@link #findMinimalUsersByNames(long, Collection)}
     * should be used instead due to performance reasons.
     *
     * @param directoryID the directory to search for the users.
     * @param usernames   names of users to find
     * @return collection of found users as InternalUsers.
     */
    Collection<InternalUser> findByNames(long directoryID, Collection<String> usernames);

    /**
     * Bulk find of users using SQL disjunction. This will return a set of minimal DTOs with just the username, id and
     * directory id.
     *
     * @param directoryId the directory to search for the users.
     * @param usernames   names of users to find
     * @return collection of found users as MinimalUsers.
     */
    Collection<MinimalUser> findMinimalUsersByNames(long directoryId, Collection<String> usernames);

    /**
     * Bulk find of users with the specified database identifiers. The result collection may be smaller than the
     * input if ids that did not match with any users were supplied
     *
     * @param userIds identifiers of users that will be found
     * @return the matching users
     */
    Collection<InternalUser> findByIds(Collection<Long> userIds);

    /**
     * Searches the specified directory for usernames of users with the specified external ids, returns
     * a map from external id to username
     *
     * @param directoryId the directory to search for the users
     * @param externalIds the external ids of the users to search for
     */
    Map<String, String> findByExternalIds(long directoryId, Set<String> externalIds);

    /**
     * Searches the specified directory for Internal Users who will be notified, that their passwords will soon expire
     *
     * @param currentTime timestamp of current time
     * @param passwordMaxChangeTime time until password will expire after updating (in days)
     * @param remindPeriod duration of remind period - user will be notified if time until his password will
     *                     expire will be lower than that time (in days)
     * @param directoryId the directory to search for the users
     * @return the matching users
     */
    Collection<InternalUserWithPasswordLastChanged> findUsersForPasswordExpiryNotification(Instant currentTime, Duration passwordMaxChangeTime,
                                                                                           Duration remindPeriod, long directoryId, int maxResults);
    /**
     * Sets the specified attribute for all specified users and after executing this,
     * every user will have exactly one attribute with that name.
     *
     * @param users users to update
     * @param attributeName name of attribute to update
     * @param attributeValue value of attribute to update
     */
    void setAttribute(Collection<InternalUser> users, String attributeName, String attributeValue);

    /**
     * Returns users with the given numeric attribute in the specified range. The results are returned in
     * ascending order by the attribute value.
     *
     * @param attributeName name of the attribute to match
     * @param min minimum numeric value of the attribute to match
     * @param max maximum numeric value of the attribute to match
     * @return matching users, in ascending order by the attribute value
     */
    List<InternalUser> findByNumericAttributeRange(String attributeName, long min, long max);
}
