package com.atlassian.crowd.dao.token;

import com.atlassian.crowd.exception.ObjectAlreadyExistsException;
import com.atlassian.crowd.model.token.ExpirableUserToken;
import com.atlassian.crowd.model.token.ExpirableUserTokenType;

import java.util.Date;
import java.util.Optional;
import java.util.Set;

/**
 * DAO for {@link ExpirableUserToken}
 *
 * @since v2.8
 */
public interface ExpirableUserTokenDao {
    /**
     * Find the token with the given random hash
     *
     * @param token a random hash
     * @return the token that was found, or none
     */
    Optional<ExpirableUserToken> findByToken(String token);

    /**
     * Persist the given token
     *
     * @param token a token
     * @return the persisted token
     * @throws ObjectAlreadyExistsException if a token with the same random hash already exists
     */
    ExpirableUserToken add(ExpirableUserToken token) throws ObjectAlreadyExistsException;

    /**
     * Removes the token with the given random hash
     *
     * @param token a random hash
     * @return true if a token was deleted, false otherwise
     */
    boolean removeByToken(String token);

    /**
     * Remove all the tokens whose expiry date is older than the given cutoff date
     *
     * @param expiryDate a date
     * @return true if some tokens were deleted, false otherwise
     */
    boolean removeExpiredTokens(Date expiryDate);

    /**
     * Remove all the tokens matching the username and directory id
     *
     * @param directoryId            directory where the user lives
     * @param username               username to remove tokens for
     * @param expirableUserTokenType token type
     * @return true if some tokens were deleted, false otherwise
     */
    boolean removeByDirectoryAndUsername(long directoryId, String username,
                                         ExpirableUserTokenType expirableUserTokenType);

    /**
     * Returns all tokens for a given user.
     *
     * @param directoryId            user's directory ID
     * @param username               username of the user
     * @param expirableUserTokenType token type
     * @return set of all tokens belonging to the user, or empty set if no tokens
     */
    Set<ExpirableUserToken> findAllTokens(long directoryId, String username,
                                          ExpirableUserTokenType expirableUserTokenType);
}
