package com.atlassian.crowd.dao.permission;


import com.atlassian.crowd.manager.permission.PermittedGroup;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.DirectoryMapping;
import com.atlassian.crowd.model.permission.InternalGrantedPermission;
import com.atlassian.crowd.model.permission.UserPermission;

import java.util.Collection;
import java.util.List;

/**
 * Manages persistence of InternalGrantedPermission at the explicit / direct permission level.
 *
 * This class does not have a concept of implied permissions (eg. doesn't know that having SYS_ADMIN implies ADMIN) and deals
 * only with direct permissions. Granting or revoking an explicit permission does not guarantee
 * a change in actual user permissions due to potential implied permissions.
 */
public interface InternalUserPermissionDAO {

    /**
     * Return true if the InternalGrantedPermission has been explicitly granted. Ie. there is a row
     * matching the group and permission specified by the permission param
     *
     * @param permission the group/permission pair to search for
     * @return true if the permission has been explicitly granted (not implicitly)
     */
    boolean exists(InternalGrantedPermission permission);

    /**
     * Remove the explicit permission from the group.
     *
     * @param permission the group/permission pair to remove
     * @return true if the permission existed and was removed
     */
    boolean revoke(InternalGrantedPermission permission);

    /**
     * Revokes all permissions associated with the specified directory mapping
     *
     * @param directoryMapping the directory mapping to revoke permissions for
     * @return the number of permissions removed
     */
    int revokeAll(DirectoryMapping directoryMapping);

    /**
     * Grant the explicit permission to the group. If the permission already explicitly exists, do nothing.
     *
     * @param permission the group/permission pair to create
     */
    void grant(InternalGrantedPermission permission);

    /**
     * Find all groups that have been explicitly (not implicitly) granted the provided permission
     *
     * @param permission permission to search for
     * @return all groups with this permission
     */
    Collection<PermittedGroup> getGrantedPermissions(UserPermission permission);

    /**
     * Find all groups that have been explicitly (not implicitly) granted the provided permission
     *
     * @param permission permission to search for
     * @param application application the permission should apply to
     * @return all groups with this permission
     */
    Collection<PermittedGroup> getGrantedPermissions(UserPermission permission, Application application);

    /**
     * Find all groups with an explicit permission, and return their highest permission
     *
     * @param prefix prefix return only groups whose names start with this prefix
     * @param start  index to start page at
     * @param limit  max number of results to return, 0 for all results
     * @return the highest permission for each group with an explicit permission ordered by groupName.
     * If a group has ADMIN and SYS_ADMIN, only return SYS_ADMIN
     */
    List<PermittedGroup> findHighestPermissionPerGroupByPrefix(String prefix, int start, int limit);

    /**
     * Find all groups with an explicit permission, and return their highest permission
     *
     * @param start index to start page at
     * @param limit max number of results to return, 0 for all results
     * @return the highest permission for each group with an explicit permission ordered by groupName.
     * If a group has ADMIN and SYS_ADMIN, only return SYS_ADMIN
     */
    List<PermittedGroup> findHighestPermissionPerGroup(int start, int limit);

    /**
     * Find all permission entries for this group. Used for eg. cleaning up when deleting the group.
     *
     * @param groupName name of group to search
     * @return database entries ordered by groupName
     */
    List<InternalGrantedPermission> findAllPermissionsForGroup(String groupName, long directoryId);
}
