package com.atlassian.crowd.dao.membership.cache;

import com.atlassian.crowd.dao.membership.InternalMembershipDao;
import com.atlassian.crowd.model.membership.InternalMembership;
import com.atlassian.crowd.util.BatchResult;

import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * Caching wrapper over {@link InternalMembershipDao}.
 */
public class CachingInternalMembershipDao extends CachingMembershipDao implements InternalMembershipDao {
    private final InternalMembershipDao delegate;

    public CachingInternalMembershipDao(InternalMembershipDao delegate,
                                        MembershipCache membershipCache) {
        super(delegate, membershipCache);
        this.delegate = delegate;
    }

    @Override
    public void removeGroupMembers(long directoryId, String groupName) {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS);
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS, groupName);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, groupName);
        delegate.removeGroupMembers(directoryId, groupName);
    }

    @Override
    public void removeGroupMemberships(long directoryId, String groupName) {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS, groupName);
        delegate.removeGroupMemberships(directoryId, groupName);
    }

    @Override
    public void removeUserMemberships(long directoryId, String username) {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS);
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, username);
        delegate.removeUserMemberships(directoryId, username);
    }

    @Override
    public void removeAllRelationships(long directoryId) {
        membershipCache.invalidateCache(directoryId);
        delegate.removeAllRelationships(directoryId);
    }

    @Override
    public void removeAllUserRelationships(long directoryId) {
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS);
        delegate.removeAllUserRelationships(directoryId);
    }

    @Override
    public void renameUserRelationships(long directoryId, String oldName, String newName) {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS);
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, oldName);
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, newName);
        delegate.renameUserRelationships(directoryId, oldName, newName);
    }

    @Override
    public void renameGroupRelationships(long directoryId, String oldName, String newName) {
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS);
        membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, oldName);
        membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, newName);
        delegate.renameGroupRelationships(directoryId, oldName, newName);
    }

    @Override
    public BatchResult<InternalMembership> addAll(Set<InternalMembership> memberships) {
        for (InternalMembership membership : memberships) {
            long directoryId = membership.getDirectory().getId();
            switch (membership.getMembershipType()) {
                case GROUP_USER:
                    membershipCache.invalidateCache(directoryId, QueryType.USER_GROUPS, membership.getLowerChildName());
                    membershipCache.invalidateCache(directoryId, QueryType.GROUP_USERS, membership.getLowerParentName());
                    break;
                case GROUP_GROUP:
                    membershipCache.invalidateCache(directoryId, QueryType.GROUP_PARENTS, membership.getLowerChildName());
                    membershipCache.invalidateCache(directoryId, QueryType.GROUP_SUBGROUPS, membership.getLowerParentName());
                    break;
            }
        }
        return delegate.addAll(memberships);
    }

    @Override
    public List<InternalMembership> getMembershipsCreatedAfter(long directoryId, Date timestamp, int maxResults) {
        return delegate.getMembershipsCreatedAfter(directoryId, timestamp, maxResults);
    }
}
