package com.atlassian.crowd.search.hibernate;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.alias.Alias;
import com.atlassian.crowd.model.application.ApplicationImpl;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.InternalGroup;
import com.atlassian.crowd.model.membership.InternalMembership;
import com.atlassian.crowd.model.membership.MembershipType;
import com.atlassian.crowd.model.token.Token;
import com.atlassian.crowd.model.user.InternalUser;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.MatchMode;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.atlassian.crowd.search.query.entity.restriction.Property;
import com.atlassian.crowd.search.query.entity.restriction.PropertyRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.AliasTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.DirectoryTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.TokenTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;
import static com.atlassian.crowd.search.Entity.ALIAS;
import static com.atlassian.crowd.search.Entity.APPLICATION;
import static com.atlassian.crowd.search.Entity.DIRECTORY;
import static com.atlassian.crowd.search.Entity.GROUP;
import static com.atlassian.crowd.search.Entity.TOKEN;
import static com.atlassian.crowd.search.Entity.USER;

/**
 * Translates implementation agnostic Queries into executable
 * Hibernate Query Language code.
 * <p>
 * Before you think this is an epic fail due the the existence of
 * Hibernate criteria queries (CBQ), criteria queries can't do
 * the join we want with user and user attribute classes without
 * explicitly mapping the join in Hibernate (AFAIK). Experience has
 * shown mapping joins for unbounded collections results in a
 * performance nightmare when mutating the collection.
 */
public class HQLQueryTranslater {
    protected static final String HQL_USER_NAME = "lowerName";
    protected static final String HQL_USER_EMAIL_ADDRESS = "lowerEmailAddress";
    protected static final String HQL_USER_FIRST_NAME = "lowerFirstName";
    protected static final String HQL_USER_LAST_NAME = "lowerLastName";
    protected static final String HQL_USER_DISPLAY_NAME = "lowerDisplayName";
    protected static final String HQL_USER_ACTIVE = "active";
    protected static final String HQL_CREATED_DATE = "createdDate";
    protected static final String HQL_UPDATED_DATE = "updatedDate";

    protected static final String HQL_GROUP_NAME = "lowerName";
    protected static final String HQL_GROUP_DESCRIPTION = "description";
    protected static final String HQL_GROUP_ACTIVE = "active";
    protected static final String HQL_GROUP_TYPE = "type";
    protected static final String HQL_GROUP_LOCAL = "local";
    protected static final String HQL_GROUP_EXTERNAL_ID = "externalId";

    protected static final String HQL_TOKEN_NAME = "name";
    protected static final String HQL_TOKEN_LAST_ACCESSED_TIME = "lastAccessedTime";
    protected static final String HQL_TOKEN_DIRECTORY_ID = "directoryId";
    protected static final String HQL_TOKEN_RANDOM_NUMBER = "randomNumber";

    protected static final String HQL_DIRECTORY_NAME = "lowerName";
    protected static final String HQL_DIRECTORY_ACTIVE = "active";
    protected static final String HQL_DIRECTORY_TYPE = "type";
    protected static final String HQL_DIRECTORY_IMPLEMENTATION_CLASS = "lowerImplementationClass";

    protected static final String HQL_APPLICATION_NAME = "lowerName";
    protected static final String HQL_APPLICATION_ACTIVE = "active";
    protected static final String HQL_APPLICATION_TYPE = "type";

    protected static final String HQL_ALIAS_NAME = "lowerAlias";
    protected static final String HQL_ALIAS_APPLICATION_ID = "application.id";
    protected static final String HQL_ALIAS_USERNAME = "lowerName";

    protected static final String HQL_ATTRIBUTE_NAME = "name";
    protected static final String HQL_ATTRIBUTE_LOWER_VALUE = "lowerValue";
    protected static final String HQL_ATTRIBUTE_NUMERIC_VALUE = "numericValue";
    protected static final String HQL_ATTRIBUTE_ALIAS = "attr";
    protected static final String HQL_DIRECTORY_ID = ".directory.id";

    protected static final String HQL_MEMBERSHIP_ALIAS = "mem";
    protected static final String HQL_MEMBERSHIP_TYPE = "membershipType";
    protected static final String HQL_MEMBERSHIP_GROUP_TYPE = "groupType";

    protected static final int DEFAULT_OR_BATCH_SIZE = 1000;
    public static final String HQL_AND = " AND ";

    private int orBatchSize;

    public HQLQueryTranslater() {
        orBatchSize = DEFAULT_OR_BATCH_SIZE;
    }

    @VisibleForTesting
    public HQLQueryTranslater(int orBatchSize) {
        this.orBatchSize = orBatchSize;
    }

    /**
     * Translates a membership query into a HQLQuery. Query parameters are translated to JPA-style
     * named parameters (e.g., :param1, :param2...).
     *
     * @param query a membership query
     * @return translated HQL query
     */
    public HQLQuery asHQL(long directoryID, MembershipQuery query) {
        return asHQL(directoryID, query, false);
    }

    /**
     * Translates a membership query into a HQLQuery. Query parameters are translated to JPA-style
     * named parameters (e.g., :param1, :param2...).
     *
     * @param query a membership query
     * @param selectEntityToMatch whether name of the entity to match should be returned, useful when there are multiple
     *                        {@link MembershipQuery#getEntityNamesToMatch()} specified.
     * @return translated HQL query
     */
    public HQLQuery asHQL(long directoryID, MembershipQuery<?> query, boolean selectEntityToMatch) {
        final HQLQuery hql = newQuery();
        hql.offsetResults(query.getStartIndex());
        hql.limitResults(query.getMaxResults());

        if (selectEntityToMatch) {
            hql.appendSelect(matchAttribute(query, false)).append(", ");
        } else {
            if (query.getEntityNamesToMatch().size() > 1) {
                hql.requireDistinct();
            }
            hql.setComparatorForBatch(NameComparator.of(query.getReturnType()));
        }

        appendWhere(query, hql, directoryID);

        final int start = selectEntityToMatch ? 1 : 0;
        if (query.getReturnType() == String.class) {
            appendAttributes(hql, ImmutableList.of(selectAttribute(query, false)), selectAttribute(query, true), start, values -> values[start]);
        } else {
            final String alias = transformEntityToAlias(query.getEntityToReturn().getEntityType());
            final String orderBy = alias + "." + resolveDefaultOrderByFieldForEntity(query.getEntityToReturn().getEntityType());
            final CustomDataFetcher<?> fetcher = CustomDataFetchers.entityProducer(query.getReturnType());
            appendAttributes(hql, fetcher.attributes(alias), orderBy, start, fetcher.getTransformer(start));
        }
        return hql;
    }

    private void appendWhere(MembershipQuery<?> query, HQLQuery hql, long directoryID) {
        if (query.getReturnType() != String.class || hasRestriction(query)) {
            final String alias = transformEntityToAlias(query.getEntityToReturn().getEntityType());
            String persistedClass = transformEntityToPersistedClass(query.getEntityToReturn().getEntityType());
            hql.appendFrom(persistedClass).append(" ").append(alias).append(", ");

            hql.safeAppendWhere(alias).safeAppendWhere(".id = ").safeAppendWhere(HQL_MEMBERSHIP_ALIAS);
            hql.safeAppendWhere(query.isFindChildren() ? ".childId" : ".parentId").safeAppendWhere(HQL_AND);
        }

        hql.appendFrom(InternalMembership.class.getSimpleName()).append(" ").append(HQL_MEMBERSHIP_ALIAS);
        String placeholder = hql.addParameterPlaceholderForBatchedParam(toLowerCase(query.getEntityNamesToMatch()));
        hql.safeAppendWhere(matchAttribute(query, true) + " in ("+ placeholder + ")");

        appendPropertyRestrictionIfNeeded(hql, query.getEntityToReturn().getEntityType(), query);

        appendMembershipTypeAndDirectoryIDAndGroupType(directoryID, query, hql);
    }

    /**
     * @return a fresh instance of a HQLQuery
     */
    protected HQLQuery newQuery() {
        return new HQLQuery();
    }

    private void appendMembershipTypeAndDirectoryIDAndGroupType(final long directoryID, final MembershipQuery query, final HQLQuery hql) {
        MembershipType membershipType;
        if (query.getEntityToMatch().getEntityType() == GROUP && query.getEntityToReturn().getEntityType() == GROUP) {
            membershipType = MembershipType.GROUP_GROUP;
        } else {
            membershipType = MembershipType.GROUP_USER;
        }
        hql.appendWhere(HQL_AND).append(HQL_MEMBERSHIP_ALIAS).append(".").append(HQL_MEMBERSHIP_TYPE).append(" = ")
                .append(hql.addParameterPlaceholder(membershipType));

        hql.appendWhere(HQL_AND).append(HQL_MEMBERSHIP_ALIAS).append(".directory.id = ")
                .append(hql.addParameterPlaceholder(directoryID));

        // add group type restriction if present
        GroupType groupType = null;
        if (query.getEntityToMatch().getEntityType() == Entity.GROUP) {
            groupType = query.getEntityToMatch().getGroupType();
        }
        if (query.getEntityToReturn().getEntityType() == Entity.GROUP) {
            if (groupType != null && groupType != query.getEntityToReturn().getGroupType()) {
                throw new IllegalArgumentException("Cannot search memberships of conflicting group types");
            }
            groupType = query.getEntityToReturn().getGroupType();
        }

        if (groupType != null) {
            hql.appendWhere(HQL_AND).append(HQL_MEMBERSHIP_ALIAS).append(".").append(HQL_MEMBERSHIP_GROUP_TYPE)
                    .append(" = ").append(hql.addParameterPlaceholder(groupType));
        }
    }

    private void appendAttributes(HQLQuery hql, List<String> select, String orderBy, int start, Function<Object[], ?> transformer) {
        final List<String> allSelect = new ArrayList<>(select);
        if (!allSelect.contains(orderBy) && !allSelect.contains(StringUtils.substringBefore(orderBy, "."))) {
            allSelect.add(orderBy);
        }
        hql.appendSelect(String.join(", ", allSelect));
        hql.appendOrderBy(orderBy);
        hql.setResultTransform(listTransformer(
                start == 0 ? transformer : values -> transform(values, start, start + allSelect.size(), transformer)));
    }

    private static Function<List<Object[]>, List<?>> listTransformer(Function<Object[], ?> transformer) {
        return results -> results.stream().map(transformer).collect(Collectors.toList());
    }

    private static Object[] transform(Object[] arr, int start, int end, Function<Object[], ?> transformer) {
        Object[] transformed = new Object[arr.length - end + start + 1];
        System.arraycopy(arr, 0, transformed, 0, start);
        transformed[start] = transformer.apply(arr);
        System.arraycopy(arr, end, transformed, start + 1, arr.length - end);
        return transformed;
    }

    private String matchAttribute(final MembershipQuery query, boolean lower) {
        return membershipAttribute(query.isFindChildren(), lower);
    }

    private String selectAttribute(final MembershipQuery query, boolean lower) {
        return membershipAttribute(!query.isFindChildren(), lower);
    }

    private String membershipAttribute(final boolean parent, boolean lower) {
        final String attribute = parent ? "parentName" : "childName";
        return HQL_MEMBERSHIP_ALIAS + "." + (lower ? "lower" + StringUtils.capitalize(attribute) : attribute);
    }

    /**
     * Translates an entity query into a HQLQuery. Query parameters are translated to JPA-style
     * positional parameters (e.g., ?1, ?2...).
     *
     * @param entityQuery an entity query
     * @return translated HQL query
     */
    public HQLQuery asHQL(EntityQuery entityQuery) {
        final HQLQuery hql = newQuery();

        appendQueryAsHQL(entityQuery, hql);

        return hql;
    }

    /**
     * Translates an entity query into a HQLQuery. Query parameters are translated to JPA-style
     * positional parameters (e.g., ?1, ?2...).
     *
     * @param entityQuery an entity query
     * @return Translated HQL query. If the result contains multiple entries it means that the original query was batched,
     * and thus start indexes and limits might have been altered for merging purposes. If the result contains single query,
     * then it's original query translated, with start index and limit preserved.
     */
    public List<HQLQuery> asHQL(long directoryID, EntityQuery entityQuery) {
        final List<EntityQuery> queries = splitEntityQueryIntoBatches(entityQuery);
        List<HQLQuery> translatedQueries = new ArrayList<>(queries.size());
        for (EntityQuery query : queries) {
            final HQLQuery hql = newQuery();

            String entityAlias = transformEntityToAlias(query.getEntityDescriptor().getEntityType());

            hql.appendWhere(entityAlias).append(HQL_DIRECTORY_ID).append(" = ")
                    .append(hql.addParameterPlaceholder(directoryID));

            appendQueryAsHQL(query, hql);
            translatedQueries.add(hql);
        }

        return translatedQueries;

    }

    private List<EntityQuery> splitEntityQueryIntoBatches(EntityQuery entityQuery) {
        if (entityQuery.getSearchRestriction() instanceof BooleanRestriction
                && ((BooleanRestriction) entityQuery.getSearchRestriction()).getBooleanLogic() == BooleanRestriction.BooleanLogic.OR) {
            final BooleanRestriction restriction = (BooleanRestriction) entityQuery.getSearchRestriction();
            final Iterable<List<SearchRestriction>> partitions = Iterables.partition(restriction.getRestrictions(), orBatchSize);

            final List<EntityQuery> queries = new ArrayList<>();
            for (List<SearchRestriction> partitionedRestrictions : partitions) {
                final EntityQuery partitionedQuery = QueryBuilder.queryFor(entityQuery.getReturnType(), entityQuery.getEntityDescriptor(),
                        Combine.anyOf(partitionedRestrictions), 0, calculateMaxResults(entityQuery));
                queries.add(partitionedQuery);
            }
            if (queries.size() > 1) {
                return queries;
            }
        }
        return Lists.newArrayList(entityQuery);
    }

    @VisibleForTesting
    int calculateMaxResults(EntityQuery entityQuery) {
        return EntityQuery.addToMaxResults(entityQuery.getMaxResults(), entityQuery.getStartIndex());
    }

    protected void appendQueryAsHQL(EntityQuery<?> query, HQLQuery hql) {
        String persistedClass = transformEntityToPersistedClass(query.getEntityDescriptor().getEntityType());
        String alias = transformEntityToAlias(query.getEntityDescriptor().getEntityType());

        hql.appendFrom(persistedClass).append(" ").append(alias);

        final String orderBy = alias + "." + resolveOrderByField(query);
        if (query.getReturnType() == String.class) {
            appendAttributes(hql, ImmutableList.of(alias + ".name"), orderBy, 0, values -> values[0]);
        } else {
            CustomDataFetcher<?> fetcher = CustomDataFetchers.entityProducer(query.getReturnType());
            appendAttributes(hql, fetcher.attributes(alias), orderBy, 0, fetcher.getTransformer(0));
        }

        // special case for GroupType restriction
        if (query.getEntityDescriptor().getEntityType() == Entity.GROUP && query.getEntityDescriptor().getGroupType() != null) {
            if (hql.whereRequired) {
                hql.appendWhere(HQL_AND);
            }
            appendGroupTypeRestrictionAsHQL(hql, query.getEntityDescriptor().getGroupType());
        }

        appendPropertyRestrictionIfNeeded(hql, query.getEntityDescriptor().getEntityType(), query);
        hql.offsetResults(query.getStartIndex());
        hql.limitResults(query.getMaxResults());
    }

    private boolean hasRestriction(Query<?> query) {
        return query.getSearchRestriction() != null && !(query.getSearchRestriction() instanceof NullRestriction);
    }

    private void appendPropertyRestrictionIfNeeded(HQLQuery hql, Entity entityType, Query<?> query) {
        if (hasRestriction(query)) {
            if (hql.whereRequired) {
                // if where was used previously we need to join to the where with AND, otherwise we can just append to the where clause
                hql.appendWhere(HQL_AND);
            }
            appendPropertyRestrictionAsHQL(hql, entityType, query.getSearchRestriction(), null);
        }
    }

    /*
     * Due to nested queries, shared alias for the attribute table are scoped to the call stack.
     */
    @SuppressWarnings("unchecked")
    protected void appendPropertyRestrictionAsHQL(HQLQuery hql, Entity entityType, SearchRestriction restriction,
                                                  @Nullable String attributeSharedAlias) {
        if (restriction instanceof PropertyRestriction) {
            final PropertyRestriction propertyRestriction = (PropertyRestriction) restriction;

            if (MatchMode.NULL == propertyRestriction.getMatchMode()) {
                appendIsNullTermRestrictionAsHSQL(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else if (String.class.equals(propertyRestriction.getProperty().getPropertyType())) {
                appendStringTermRestrictionAsHQL(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else if (Boolean.class.equals(propertyRestriction.getProperty().getPropertyType())) {
                appendBooleanTermRestrictionAsHQL(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else if (Enum.class.isAssignableFrom(propertyRestriction.getProperty().getPropertyType())) {
                appendEnumTermRestrictionAsHQL(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else if (Date.class.isAssignableFrom(propertyRestriction.getProperty().getPropertyType())) {
                appendDateTermRestriction(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else if (Number.class.isAssignableFrom(propertyRestriction.getProperty().getPropertyType())) {
                appendNumberTermRestriction(hql, entityType, propertyRestriction, attributeSharedAlias);
            } else {
                throw new IllegalArgumentException("ProperyRestriction unsupported: " + restriction.getClass());
            }
        } else if (restriction instanceof BooleanRestriction) {
            appendMultiTermRestrictionAsHQL(hql, entityType, (BooleanRestriction) restriction);
        } else {
            throw new IllegalArgumentException("ProperyRestriction unsupported: " + restriction.getClass());
        }
    }

    protected void appendIsNullTermRestrictionAsHSQL(final HQLQuery hql, final Entity entityType, final PropertyRestriction<?> restriction,
                                                     final @Nullable String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        hql.appendWhere("IS NULL");
    }

    private void appendNumberTermRestriction(final HQLQuery hql, final Entity entityType, final PropertyRestriction<? extends Number> restriction,
                                             final @Nullable String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        appendComparableValueAsHQL(hql, restriction);
    }

    protected void appendDateTermRestriction(final HQLQuery hql, final Entity entityType, final PropertyRestriction<? extends Date> restriction,
                                             final @Nullable String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        appendComparableValueAsHQL(hql, restriction);
    }

    protected void appendBooleanTermRestrictionAsHQL(final HQLQuery hql, final Entity entityType, final PropertyRestriction<Boolean> restriction,
                                                     final @Nullable String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        hql.appendWhere("= ").append(hql.addParameterPlaceholder(restriction.getValue()));
    }

    protected void appendEnumTermRestrictionAsHQL(final HQLQuery hql, final Entity entityType, final PropertyRestriction<Enum> restriction,
                                                  final @Nullable String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        hql.appendWhere("= ").append(hql.addParameterPlaceholder(restriction.getValue()));
    }

    protected void appendMultiTermRestrictionAsHQL(HQLQuery hql, Entity entityType, BooleanRestriction booleanRestriction) {
        String attributeSharedAlias = getAttributeSharedAlias(hql, entityType, booleanRestriction);

        appendBooleanLogicWhereClause(booleanRestriction.getRestrictions(), hql, booleanRestriction.getBooleanLogic(), (restriction) -> {
            appendPropertyRestrictionAsHQL(hql, entityType, restriction, attributeSharedAlias);
        });
    }

    private String resolveBooleanOperator(BooleanRestriction.BooleanLogic booleanLogic) {
        switch (booleanLogic) {
            case AND:
                return HQL_AND;
            case OR:
                return " OR ";
            default:
                throw new IllegalArgumentException("BooleanLogic unsupported: " + booleanLogic);
        }
    }

    private <T> void appendBooleanLogicWhereClause(Iterable<T> restrictions, HQLQuery hqlQuery, BooleanRestriction.BooleanLogic logic, Consumer<T> partitionConsumer) {
        final String booleanOperator = resolveBooleanOperator(logic);
        hqlQuery.appendWhere("(");
        final Iterator<T> iterator = restrictions.iterator();
        while (iterator.hasNext()) {
            partitionConsumer.accept(iterator.next());
            if (iterator.hasNext()) {
                hqlQuery.appendWhere(booleanOperator);
            }
        }
        hqlQuery.appendWhere(")");
    }


    /**
     * If possible, prepares a new attribute alias that can be shared by the sub-restrictions of a boolean restriction.
     *
     * @param hql                the HQL query for which the attribute will be preparated
     * @param entityType         entity type the attribute is associated with (e.g, user, group)
     * @param booleanRestriction boolean restriction which sub-restrictions may share an attribute alias
     * @return the alias for the attribute, or <code>null</code> if sharing is not possible, i.e., each sub-restriction
     * must use its own alias.
     */
    @VisibleForTesting
    @Nullable
    static String getAttributeSharedAlias(HQLQuery hql, Entity entityType, BooleanRestriction booleanRestriction) {
        if (booleanRestriction.getBooleanLogic() == BooleanRestriction.BooleanLogic.OR) {
            Optional<EntityJoiner> joiner = EntityJoiner.forEntity(entityType);
            if (joiner.isPresent()) {
                return joiner.get().leftJoinAttributesIfSecondary(hql, booleanRestriction);
            }
        }

        return null;
    }

    /**
     * @param primaryProperties Set of primary properties.
     * @return a Predicate that returns <code>true</code> for secondary property restrictions.
     * @see UserTermKeys#ALL_USER_PROPERTIES
     * @see GroupTermKeys#ALL_GROUP_PROPERTIES
     */
    @VisibleForTesting
    static Predicate<SearchRestriction> isSecondaryPropertyRestriction(final Set<Property<?>> primaryProperties) {
        return searchRestriction -> searchRestriction instanceof PropertyRestriction
                && !primaryProperties.contains(((PropertyRestriction) searchRestriction).getProperty());
    }

    protected void appendStringTermRestrictionAsHQL(HQLQuery hql, Entity entityType, PropertyRestriction<String> restriction,
                                                    @Nullable final String attributeSharedAlias) {
        appendEntityPropertyAsHQL(hql, entityType, restriction, attributeSharedAlias);
        appendStringValueAsHQL(hql, restriction);
    }

    protected void appendEntityPropertyAsHQL(final HQLQuery hql, Entity entityType, final PropertyRestriction restriction,
                                             @Nullable final String attributeSharedAlias) {
        switch (entityType) {
            case USER:
                appendUserPropertyAsHQL(hql, restriction, attributeSharedAlias);
                break;
            case GROUP:
                appendGroupPropertyAsHQL(hql, restriction, attributeSharedAlias);
                break;
            case TOKEN:
                appendTokenPropertyAsHQL(hql, restriction);
                break;
            case DIRECTORY:
                appendDirectoryPropertyAsHQL(hql, restriction);
                break;
            case APPLICATION:
                appendApplicationPropertyAsHQL(hql, restriction);
                break;
            case ALIAS:
                appendAliasPropertyAsHQL(hql, restriction);
                break;
            default:
                throw new IllegalArgumentException("Cannot form property restriction for entity of type <" + entityType + ">");
        }
    }

    private void appendAliasPropertyAsHQL(final HQLQuery hql, final PropertyRestriction restriction) {
        String alias = transformEntityToAlias(ALIAS);

        if (restriction.getProperty().equals(AliasTermKeys.ALIAS)) {
            hql.appendWhere(alias).append(".").append(HQL_ALIAS_NAME);
        } else if (restriction.getProperty().equals(AliasTermKeys.APPLICATION_ID)) {
            hql.appendWhere(alias).append(".").append(HQL_ALIAS_APPLICATION_ID);
        } else {
            throw new IllegalArgumentException("Alias does not support searching by property: " + restriction.getProperty().getPropertyName());
        }

        hql.appendWhere(" ");

    }

    private void appendApplicationPropertyAsHQL(final HQLQuery hql, final PropertyRestriction restriction) {
        String alias = transformEntityToAlias(APPLICATION);

        if (restriction.getProperty().equals(DirectoryTermKeys.NAME)) {
            hql.appendWhere(alias).append(".").append(HQL_APPLICATION_NAME);
        } else if (restriction.getProperty().equals(DirectoryTermKeys.ACTIVE)) {
            hql.appendWhere(alias).append(".").append(HQL_APPLICATION_ACTIVE);
        } else if (restriction.getProperty().equals(DirectoryTermKeys.TYPE)) {
            hql.appendWhere(alias).append(".").append(HQL_APPLICATION_TYPE);
        } else {
            throw new IllegalArgumentException("Application does not support searching by property: " + restriction.getProperty().getPropertyName());
        }

        hql.appendWhere(" ");
    }

    protected void appendDirectoryPropertyAsHQL(final HQLQuery hql, final PropertyRestriction restriction) {
        String alias = transformEntityToAlias(DIRECTORY);

        if (restriction.getProperty().equals(DirectoryTermKeys.NAME)) {
            hql.appendWhere(alias).append(".").append(HQL_DIRECTORY_NAME);
        } else if (restriction.getProperty().equals(DirectoryTermKeys.ACTIVE)) {
            hql.appendWhere(alias).append(".").append(HQL_DIRECTORY_ACTIVE);
        } else if (restriction.getProperty().equals(DirectoryTermKeys.IMPLEMENTATION_CLASS)) {
            hql.appendWhere(alias).append(".").append(HQL_DIRECTORY_IMPLEMENTATION_CLASS);
        } else if (restriction.getProperty().equals(DirectoryTermKeys.TYPE)) {
            hql.appendWhere(alias).append(".").append(HQL_DIRECTORY_TYPE);
        } else {
            throw new IllegalArgumentException("Directory does not support searching by property: " + restriction.getProperty().getPropertyName());
        }

        hql.appendWhere(" ");
    }

    protected void appendTokenPropertyAsHQL(final HQLQuery hql, final PropertyRestriction restriction) {
        String tokenAlias = transformEntityToAlias(TOKEN);

        if (restriction.getProperty().equals(TokenTermKeys.NAME)) {
            hql.appendWhere(tokenAlias).append(".").append(HQL_TOKEN_NAME);
        } else if (restriction.getProperty().equals(TokenTermKeys.LAST_ACCESSED_TIME)) {
            hql.appendWhere(tokenAlias).append(".").append(HQL_TOKEN_LAST_ACCESSED_TIME);
        } else if (restriction.getProperty().equals(TokenTermKeys.DIRECTORY_ID)) {
            hql.appendWhere(tokenAlias).append(".").append(HQL_TOKEN_DIRECTORY_ID);
        } else if (restriction.getProperty().equals(TokenTermKeys.RANDOM_NUMBER)) {
            hql.appendWhere(tokenAlias).append(".").append(HQL_TOKEN_RANDOM_NUMBER);
        } else {
            throw new IllegalArgumentException("Token does not support searching by property: " + restriction.getProperty().getPropertyName());
        }

        hql.appendWhere(" ");
    }

    protected void appendGroupTypeRestrictionAsHQL(final HQLQuery hql, final GroupType groupType) {
        if (groupType != null) {
            String groupAlias = transformEntityToAlias(GROUP);
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_TYPE);
            hql.appendWhere(" = ").append(hql.addParameterPlaceholder(groupType));
        }
    }

    protected void appendGroupPropertyAsHQL(final HQLQuery hql, final PropertyRestriction restriction,
                                            @Nullable final String attributeSharedAlias) {
        String groupAlias = transformEntityToAlias(GROUP);

        if (restriction.getProperty().equals(GroupTermKeys.NAME)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_NAME);
        } else if (restriction.getProperty().equals(GroupTermKeys.DESCRIPTION)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_DESCRIPTION);
        } else if (restriction.getProperty().equals(GroupTermKeys.ACTIVE)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_ACTIVE);
        } else if (restriction.getProperty().equals(GroupTermKeys.CREATED_DATE)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_CREATED_DATE);
        } else if (restriction.getProperty().equals(GroupTermKeys.UPDATED_DATE)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_UPDATED_DATE);
        } else if (restriction.getProperty().equals(GroupTermKeys.LOCAL)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_LOCAL);
        } else if (restriction.getProperty().equals(GroupTermKeys.EXTERNAL_ID)) {
            hql.appendWhere(groupAlias).append(".").append(HQL_GROUP_EXTERNAL_ID);
        } else {
            // custom attribute
            if (restriction.getMatchMode() == MatchMode.NULL) {
                String attrAlias = HQL_ATTRIBUTE_ALIAS + hql.getNextAlias();

                hql.appendWhere("NOT EXISTS (SELECT 1")
                        .append(" FROM InternalGroupAttribute ").append(attrAlias)
                        .append(" WHERE ").append(groupAlias).append(".id = ").append(attrAlias).append(".group.id")
                        .append(HQL_AND).append(attrAlias).append(".").append(HQL_ATTRIBUTE_NAME).append(" = ")
                        .append(hql.addParameterPlaceholder(restriction.getProperty().getPropertyName()))
                        .append(")");

                // needed because "IS NULL" will be appended later, and we need to ignore it.
                hql.appendWhere(HQL_AND).append(hql.addParameterPlaceholder(null));
            } else {
                String attrAlias;

                if (attributeSharedAlias == null) {
                    attrAlias = EntityJoiner.GROUP.leftJoinAttributes(hql);
                } else {
                    attrAlias = attributeSharedAlias;
                }

                hql.appendWhere(groupAlias).append(".id = ").append(attrAlias).append(".group.id").append(HQL_AND)
                        .append(attrAlias).append(".").append(HQL_ATTRIBUTE_NAME).append(" = ")
                        .append(hql.addParameterPlaceholder(restriction.getProperty().getPropertyName()))
                        .append(HQL_AND).append(attrAlias).append(".").append(HQL_ATTRIBUTE_LOWER_VALUE);
            }

            hql.requireDistinct(); // needed because a join across the tables could potentially produce duplicate results in result set (think: multiply correct disjuction)
        }
        hql.appendWhere(" ");
    }

    protected void appendUserPropertyAsHQL(HQLQuery hql, PropertyRestriction restriction,
                                           @Nullable final String attributeSharedAlias) {
        String userAlias = transformEntityToAlias(USER);

        if (restriction.getProperty().equals(UserTermKeys.USERNAME)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_NAME);
        } else if (restriction.getProperty().equals(UserTermKeys.EMAIL)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_EMAIL_ADDRESS);
        } else if (restriction.getProperty().equals(UserTermKeys.FIRST_NAME)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_FIRST_NAME);
        } else if (restriction.getProperty().equals(UserTermKeys.LAST_NAME)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_LAST_NAME);
        } else if (restriction.getProperty().equals(UserTermKeys.DISPLAY_NAME)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_DISPLAY_NAME);
        } else if (restriction.getProperty().equals(UserTermKeys.ACTIVE)) {
            hql.appendWhere(userAlias).append(".").append(HQL_USER_ACTIVE);
        } else if (restriction.getProperty().equals(UserTermKeys.CREATED_DATE)) {
            hql.appendWhere(userAlias).append(".").append(HQL_CREATED_DATE);
        } else if (restriction.getProperty().equals(UserTermKeys.UPDATED_DATE)) {
            hql.appendWhere(userAlias).append(".").append(HQL_UPDATED_DATE);
        } else {
            // custom attribute
            if (restriction.getMatchMode() == MatchMode.NULL) {
                String attrAlias = HQL_ATTRIBUTE_ALIAS + hql.getNextAlias();

                hql.appendWhere("NOT EXISTS (SELECT 1")
                        .append(" FROM InternalUserAttribute ").append(attrAlias)
                        .append(" WHERE ").append(userAlias).append(".id = ").append(attrAlias).append(".user.id")
                        .append(HQL_AND).append(attrAlias).append(".").append(HQL_ATTRIBUTE_NAME).append(" = ")
                        .append(hql.addParameterPlaceholder(restriction.getProperty().getPropertyName()))
                        .append(")");

                // needed because "IS NULL" will be appended later, and we need to ignore it.
                hql.appendWhere(HQL_AND).append(hql.addParameterPlaceholder(null));
            } else {
                String attrAlias;

                if (attributeSharedAlias == null) {
                    attrAlias = EntityJoiner.USER.leftJoinAttributes(hql);
                } else {
                    attrAlias = attributeSharedAlias;
                }

                final String attributeValueType = Number.class.isAssignableFrom(restriction.getProperty().getPropertyType())
                        ? HQL_ATTRIBUTE_NUMERIC_VALUE
                        : HQL_ATTRIBUTE_LOWER_VALUE;

                hql.appendWhere(userAlias).append(".id = ").append(attrAlias).append(".user.id").append(HQL_AND)
                        .append(attrAlias).append(".").append(HQL_ATTRIBUTE_NAME).append(" = ")
                        .append(hql.addParameterPlaceholder(restriction.getProperty().getPropertyName()))
                        .append(HQL_AND).append(attrAlias).append(".").append(attributeValueType);
            }

            hql.requireDistinct(); // needed because a join across the tables could potentially produce duplicate results in result set (think: multiply correct disjuction)
        }
        hql.appendWhere(" ");
    }

    protected void appendStringValueAsHQL(HQLQuery hql, PropertyRestriction<String> restriction) {
        final String value = isCaseSensitiveProperty(restriction.getProperty()) ?
                toLowerCase(restriction.getValue()) : restriction.getValue();

        switch (restriction.getMatchMode()) {
            case STARTS_WITH:
                hql.appendWhere("LIKE ").append(hql.addParameterPlaceholder(value + "%"));
                break;
            case ENDS_WITH:
                hql.appendWhere("LIKE ").append(hql.addParameterPlaceholder("%" + value));
                break;
            case CONTAINS:
                hql.appendWhere("LIKE ").append(hql.addParameterPlaceholder("%" + value + "%"));
                break;
            default:
                hql.appendWhere("= ").append(hql.addParameterPlaceholder(value));
        }
    }

    /**
     * @param property the property to check for case sensitivity
     * @return <code>true</code> if values of this property are in lowercase
     */
    @VisibleForTesting
    static boolean isCaseSensitiveProperty(Property<String> property) {
        return !property.equals(GroupTermKeys.DESCRIPTION)
                && !UserTermKeys.EXTERNAL_ID.equals(property);
    }

    protected void appendComparableValueAsHQL(HQLQuery hql, PropertyRestriction restriction) {
        switch (restriction.getMatchMode()) {
            case GREATER_THAN:
                hql.appendWhere("> ").append(hql.addParameterPlaceholder(restriction.getValue()));
                break;
            case GREATER_THAN_OR_EQUAL:
                hql.appendWhere(">= ").append(hql.addParameterPlaceholder(restriction.getValue()));
                break;
            case LESS_THAN:
                hql.appendWhere("< ").append(hql.addParameterPlaceholder(restriction.getValue()));
                break;
            case LESS_THAN_OR_EQUAL:
                hql.appendWhere("<= ").append(hql.addParameterPlaceholder(restriction.getValue()));
                break;
            default:
                hql.appendWhere(" = ").append(hql.addParameterPlaceholder(restriction.getValue()));
        }
    }

    private static String transformEntityToAlias(Entity entity) {
        switch (entity) {
            case USER:
                return "usr";
            case GROUP:
                return "grp";
            case TOKEN:
                return "token";
            case DIRECTORY:
                return "directory";
            case APPLICATION:
                return "application";
            case ALIAS:
                return "alias";
            default:
                throw new IllegalArgumentException("Cannot transform entity of type <" + entity + ">");
        }
    }

    private String transformEntityToPersistedClass(Entity entity) {
        switch (entity) {
            case USER:
                return InternalUser.class.getSimpleName();
            case GROUP:
                return InternalGroup.class.getSimpleName();
            case TOKEN:
                return Token.class.getSimpleName();
            case DIRECTORY:
                return DirectoryImpl.class.getSimpleName();
            case APPLICATION:
                return ApplicationImpl.class.getSimpleName();
            case ALIAS:
                return Alias.class.getSimpleName();
            default:
                throw new IllegalArgumentException("Cannot transform entity of type <" + entity + ">");
        }
    }

    private String resolveOrderByField(EntityQuery query) {
        return resolveDefaultOrderByFieldForEntity(query.getEntityDescriptor().getEntityType());
    }

    private String resolveDefaultOrderByFieldForEntity(Entity entity) {
        switch (entity) {
            case USER:
                return HQL_USER_NAME;
            case GROUP:
                return HQL_GROUP_NAME;
            case TOKEN:
                return HQL_TOKEN_NAME;
            case DIRECTORY:
                return HQL_DIRECTORY_NAME;
            case APPLICATION:
                return HQL_APPLICATION_NAME;
            case ALIAS:
                return HQL_ALIAS_USERNAME;
            default:
                throw new IllegalArgumentException("Cannot transform entity of type <" + entity + ">");
        }
    }

    @VisibleForTesting
    public void setOrBatchSize(int orBatchSize) {
        this.orBatchSize = orBatchSize;
    }

    public int getOrBatchSize() {
        return orBatchSize;
    }

    /**
     * Strategies for joining an entity with its attributes collection.
     */
    private enum EntityJoiner {
        /**
         * Joiner for User entities.
         */
        USER(Entity.USER, UserTermKeys.ALL_USER_PROPERTIES, "attributes"),

        /**
         * Joiner for Group entities
         */
        GROUP(Entity.GROUP, GroupTermKeys.ALL_GROUP_PROPERTIES, "attributes");

        /**
         * @param entity an Entity
         * @return a LeftJoiner for the given Entity or null if there isn't one.
         */
        public static Optional<EntityJoiner> forEntity(Entity entity) {
            return Optional.ofNullable(BY_ENTITY.get(entity));
        }

        /**
         * Joiners mapped by their entity.
         */
        private static final ImmutableMap<Entity, EntityJoiner> BY_ENTITY = Maps.uniqueIndex(EnumSet.allOf(EntityJoiner.class), input -> input.entity);

        private final Entity entity;
        private final Set<Property<?>> allProperties;
        private final String tableName;

        EntityJoiner(Entity entity, Set<Property<?>> allProperties, String tableName) {
            this.entity = Preconditions.checkNotNull(entity);
            this.allProperties = Preconditions.checkNotNull(allProperties);
            this.tableName = Preconditions.checkNotNull(tableName);
        }

        /**
         * Appends a <a href="http://docs.jboss.org/hibernate/orm/3.3/reference/en/html/queryhql.html#queryhql-joins">left
         * outer join</a> to the given HQL query.
         *
         * @param hql a HQLQuery
         * @return the alias for the joined attributes
         */
        public String leftJoinAttributes(HQLQuery hql) {
            String attributeAlias = HQL_ATTRIBUTE_ALIAS + hql.getNextAlias();
            hql.appendFrom(String.format(" LEFT JOIN %s.%s AS %s", transformEntityToAlias(entity), tableName, attributeAlias));

            return attributeAlias;
        }

        /**
         * Appends a <a href="http://docs.jboss.org/hibernate/orm/3.3/reference/en/html/queryhql.html#queryhql-joins">left
         * outer join</a> to the given HQL query if any of the given restrictions is on a secondary attribute.
         *
         * @param hql a HQLQuery
         * @return the alias for the joined attributes
         * @see #leftJoinAttributes(HQLQuery)
         */
        public String leftJoinAttributesIfSecondary(HQLQuery hql, BooleanRestriction booleanRestriction) {
            if (booleanRestriction.getRestrictions().stream().anyMatch(isSecondaryPropertyRestriction(allProperties))) {
                return leftJoinAttributes(hql);
            }

            return null;
        }
    }
}
