package com.atlassian.crowd.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.function.UnaryOperator;

/** Immutable implementation of {@link com.atlassian.crowd.embedded.api.Directory}, which lazily evaluates specific attributes
 */
public class LazyAttributesEvaluationDirectory extends DirectoryWrapper implements Directory {
    private final Set<String> lazyEvaluatedKeys;
    private final UnaryOperator<String> transformer;
    private final Cache<String, Optional<String>> evaluatedAttributes;

    public LazyAttributesEvaluationDirectory(Directory delegate, Set<String> lazyEvaluatedKeys, UnaryOperator<String> transformer) {
        super(delegate);
        this.lazyEvaluatedKeys = ImmutableSet.copyOf(lazyEvaluatedKeys);
        this.transformer = transformer;
        this.evaluatedAttributes = CacheBuilder.newBuilder().build();
    }

    @Nullable
    @Override
    public Set<String> getValues(String key) {
        if (lazyEvaluatedKeys.contains(key)) {
            try {
                return evaluatedAttributes.get(key, extractFromDelegateAndTransform(key))
                        .map(Collections::singleton)
                        .orElse(null);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
        return delegate.getValues(key);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        if (lazyEvaluatedKeys.contains(key)) {
            try {
                return evaluatedAttributes.get(key, extractFromDelegateAndTransform(key)).orElse(null);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
        return delegate.getValue(key);
    }

    @Override
    public Map<String, String> getAttributes() {
        final Map<String, String> attributes = new HashMap<>(delegate.getAttributes());
        lazyEvaluatedKeys.stream()
                .filter(attributes::containsKey)
                .forEach(k -> attributes.put(k, getValue(k)));
        return attributes;
    }

    private Callable<Optional<String>> extractFromDelegateAndTransform(String key) {
        return () -> {
            final Set<String> originalValues = delegate.getValues(key);
            if (originalValues == null || originalValues.isEmpty()) {
                return Optional.empty();
            }
            final String value = Iterables.getOnlyElement(originalValues);
            return Optional.of(transformer.apply(value));
        };
    }
}
