package com.atlassian.crowd.directory;

import com.atlassian.crowd.embedded.api.PasswordConstraint;
import com.atlassian.crowd.embedded.api.PasswordScore;
import com.atlassian.crowd.embedded.api.PasswordScoreService;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.api.ValidatePasswordRequest;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

public class PasswordScoreConstraint implements PasswordConstraint {
    private static final String WORD_BOUNDARY_REGEX = "(?=\\b)";

    private final PasswordScore minimumPasswordScore;
    private final PasswordScoreService passwordScoreService;

    /**
     * @param passwordScore        a minimum calculated score the password must match
     * @param passwordScoreService A service to calculate the score of a {@link com.atlassian.crowd.embedded.api.PasswordCredential}
     */
    public PasswordScoreConstraint(PasswordScore passwordScore, PasswordScoreService passwordScoreService) {
        this.minimumPasswordScore = checkNotNull(passwordScore);
        this.passwordScoreService = checkNotNull(passwordScoreService);
    }

    @Override
    public boolean validate(ValidatePasswordRequest request) {
        PasswordScore actualPasswordScore = passwordScoreService.getPasswordScore(request.getPassword(), getUserInfo(request));
        return actualPasswordScore.isAtLeast(minimumPasswordScore);
    }

    public PasswordScore getMinimumPasswordScore() {
        return minimumPasswordScore;
    }

    @Override
    public String toString() {
        return "PasswordScoreConstraint(minimum=" + minimumPasswordScore + ")";
    }

    /**
     * Creates a collection of user-specific inputs for the user in the given ValidatePasswordRequest. This includes:
     * <ul>
     * <li>the username,
     * <li>the display name, and
     * <li>the email address
     * </ul>
     * Each of these elements is tokenised on word boundaries.
     *
     * @param request a ValidatePasswordRequest
     * @return a list of score words
     */
    private Collection<String> getUserInfo(ValidatePasswordRequest request) {
        User user = request.getUser();

        return ImmutableList.<String>builder()
                .addAll(splitOnWordBoundary(user.getName()))
                .addAll(splitOnWordBoundary(user.getDisplayName()))
                .addAll(splitOnWordBoundary(user.getEmailAddress()))
                .build();
    }

    private List<String> splitOnWordBoundary(String string) {
        if (StringUtils.isBlank(string)) {
            return ImmutableList.of();
        }

        return ImmutableList.copyOf(string.split(WORD_BOUNDARY_REGEX));
    }
}
