package com.atlassian.crowd.directory;

import java.util.Set;

import javax.annotation.Nullable;

import com.atlassian.crowd.embedded.api.Attributes;
import com.atlassian.crowd.embedded.api.PasswordConstraint;
import com.atlassian.crowd.embedded.api.PasswordScore;
import com.atlassian.crowd.embedded.api.PasswordScoreService;

import com.google.common.collect.ImmutableSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.crowd.directory.AbstractInternalDirectory.ATTRIBUTE_PASSWORD_MINIMUM_LENGTH;
import static com.atlassian.crowd.directory.AbstractInternalDirectory.ATTRIBUTE_PASSWORD_MINIMUM_SCORE;
import static com.atlassian.crowd.directory.AbstractInternalDirectory.ATTRIBUTE_PASSWORD_REGEX;

/**
 * Instantiates the active {@link com.atlassian.crowd.embedded.api.PasswordConstraint}'s
 * for a directory based on the directory's attributes. This implementation uses attributes
 * defined on an {@link com.atlassian.crowd.directory.InternalDirectory}.
 *
 * @see com.atlassian.crowd.directory.InternalDirectory#ATTRIBUTE_PASSWORD_MINIMUM_LENGTH
 * @see com.atlassian.crowd.directory.InternalDirectory#ATTRIBUTE_PASSWORD_MINIMUM_SCORE
 * @see com.atlassian.crowd.directory.InternalDirectory#ATTRIBUTE_PASSWORD_REGEX
 */
public class PasswordConstraintsLoaderImpl implements PasswordConstraintsLoader {
    private static final Logger logger = LoggerFactory.getLogger(PasswordConstraintsLoaderImpl.class);

    private final PasswordScoreService passwordScoreService;

    public PasswordConstraintsLoaderImpl(PasswordScoreService passwordScoreService) {
        this.passwordScoreService = passwordScoreService;
    }

    @Override
    public Set<PasswordConstraint> getFromDirectoryAttributes(long directoryId, Attributes attributes) {
        ImmutableSet.Builder<PasswordConstraint> setBuilder = ImmutableSet.builder();

        String regex = attributes.getValue(ATTRIBUTE_PASSWORD_REGEX);
        if (regex != null) {
            setBuilder.add(new RegexConstraint(regex));
        }

        PasswordScore minimumPasswordScore = getMinimumPasswordScore(directoryId, attributes);
        if (minimumPasswordScore != null) {
            setBuilder.add(new PasswordScoreConstraint(minimumPasswordScore, passwordScoreService));
        }

        String minLength = attributes.getValue(ATTRIBUTE_PASSWORD_MINIMUM_LENGTH);
        if (minLength != null) {
            setBuilder.add(new PasswordLengthConstraint(Integer.parseInt(minLength)));
        }

        return setBuilder.build();
    }

    @Nullable
    private PasswordScore getMinimumPasswordScore(long directoryId, final Attributes attributes) {
        final String scoreRanking = attributes.getValue(ATTRIBUTE_PASSWORD_MINIMUM_SCORE);
        if (scoreRanking == null) {
            return null;
        } else {
            try {
                return PasswordScore.fromRanking(Long.parseLong(scoreRanking));
            } catch (IllegalArgumentException e) {
                logger.error("An invalid ranking for password score of {} was found in the database for directory {}", scoreRanking, directoryId);
                return null;
            }
        }
    }
}
