package com.atlassian.crowd.directory;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BoundedCount;

/**
 * A utility class which delegates (i.e. forwards) its method calls to an underlying
 * {@link #getDelegate() delegate directory}
 * <p>
 * Concrete implementations can override any method requiring custom logic to the underlying directory
 *
 * @since v2.9
 */
@ParametersAreNonnullByDefault
public abstract class AbstractForwardingDirectory implements RemoteDirectory {

    @Override
    public long getDirectoryId() {
        return getDelegate().getDirectoryId();
    }

    @Override
    public void setDirectoryId(long directoryId) {
        getDelegate().setDirectoryId(directoryId);
    }

    @Override
    @Nonnull
    public String getDescriptiveName() {
        return getDelegate().getDescriptiveName();
    }

    @Override
    public void setAttributes(Map<String, String> attributes) {
        getDelegate().setAttributes(attributes);
    }

    @Override
    @Nonnull
    public User findUserByName(String name) throws UserNotFoundException, OperationFailedException {
        return getDelegate().findUserByName(name);
    }

    @Override
    @Nonnull
    public UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException {
        return getDelegate().findUserWithAttributesByName(name);
    }

    @Override
    @Nonnull
    public User findUserByExternalId(String externalId) throws UserNotFoundException, OperationFailedException {
        return getDelegate().findUserByExternalId(externalId);
    }

    @Override
    @Nonnull
    public User authenticate(String name, PasswordCredential credential) throws UserNotFoundException, InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, OperationFailedException {
        return getDelegate().authenticate(name, credential);
    }

    @Override
    @Nonnull
    public User addUser(UserTemplate user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        return addUser(UserTemplateWithAttributes.toUserWithNoAttributes(user), credential);
    }

    @Override
    public UserWithAttributes addUser(UserTemplateWithAttributes user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        return getDelegate().addUser(user, credential);
    }

    @Override
    @Nonnull
    public User updateUser(UserTemplate user) throws InvalidUserException, UserNotFoundException, OperationFailedException {
        return getDelegate().updateUser(user);
    }

    @Override
    public void updateUserCredential(String username, PasswordCredential credential) throws UserNotFoundException, InvalidCredentialException, OperationFailedException {
        getDelegate().updateUserCredential(username, credential);
    }

    @Override
    @Nonnull
    public User renameUser(String oldName, String newName) throws UserNotFoundException, InvalidUserException, UserAlreadyExistsException, OperationFailedException {
        return getDelegate().renameUser(oldName, newName);
    }

    @Override
    public void storeUserAttributes(String username, Map<String, Set<String>> attributes) throws UserNotFoundException, OperationFailedException {
        getDelegate().storeUserAttributes(username, attributes);
    }

    @Override
    public void removeUserAttributes(String username, String attributeName) throws UserNotFoundException, OperationFailedException {
        getDelegate().removeUserAttributes(username, attributeName);
    }

    @Override
    public void removeUser(String name) throws UserNotFoundException, OperationFailedException {
        getDelegate().removeUser(name);
    }

    @Override
    @Nonnull
    public <T> List<T> searchUsers(EntityQuery<T> query) throws OperationFailedException {
        return getDelegate().searchUsers(query);
    }

    @Override
    @Nonnull
    public Group findGroupByName(String name) throws GroupNotFoundException, OperationFailedException {
        return getDelegate().findGroupByName(name);
    }

    @Override
    @Nonnull
    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException {
        return getDelegate().findGroupWithAttributesByName(name);
    }

    @Override
    @Nonnull
    public Group addGroup(GroupTemplate group) throws InvalidGroupException, OperationFailedException {
        return getDelegate().addGroup(group);
    }

    @Override
    @Nonnull
    public Group updateGroup(GroupTemplate group) throws InvalidGroupException, GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        return getDelegate().updateGroup(group);
    }

    @Override
    @Nonnull
    public Group renameGroup(String oldName, String newName) throws GroupNotFoundException, InvalidGroupException, OperationFailedException {
        return getDelegate().renameGroup(oldName, newName);
    }

    @Override
    public void storeGroupAttributes(String groupName, Map<String, Set<String>> attributes) throws GroupNotFoundException, OperationFailedException {
        getDelegate().storeGroupAttributes(groupName, attributes);
    }

    @Override
    public void removeGroupAttributes(String groupName, String attributeName) throws GroupNotFoundException, OperationFailedException {
        getDelegate().removeGroupAttributes(groupName, attributeName);
    }

    @Override
    public void removeGroup(String name) throws GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        getDelegate().removeGroup(name);
    }

    @Override
    @Nonnull
    public <T> List<T> searchGroups(EntityQuery<T> query) throws OperationFailedException {
        return getDelegate().searchGroups(query);
    }

    @Override
    public boolean isUserDirectGroupMember(String username, String groupName) throws OperationFailedException {
        return getDelegate().isUserDirectGroupMember(username, groupName);
    }

    @Override
    public boolean isGroupDirectGroupMember(String childGroup, String parentGroup) throws OperationFailedException {
        return getDelegate().isGroupDirectGroupMember(childGroup, parentGroup);
    }

    @Override
    @Nonnull
    public BoundedCount countDirectMembersOfGroup(String groupName, int querySizeHint) throws OperationFailedException {
        return getDelegate().countDirectMembersOfGroup(groupName, querySizeHint);
    }

    @Override
    public void addUserToGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        getDelegate().addUserToGroup(username, groupName);
    }

    @Override
    public void addGroupToGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        getDelegate().addGroupToGroup(childGroup, parentGroup);
    }

    @Override
    public void removeUserFromGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        getDelegate().removeUserFromGroup(username, groupName);
    }

    @Override
    public void removeGroupFromGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        getDelegate().removeGroupFromGroup(childGroup, parentGroup);
    }

    @Override
    @Nonnull
    public <T> List<T> searchGroupRelationships(MembershipQuery<T> query) throws OperationFailedException {
        return getDelegate().searchGroupRelationships(query);
    }

    @Override
    public void testConnection() throws OperationFailedException {
        getDelegate().testConnection();
    }

    @Override
    public boolean supportsInactiveAccounts() {
        return getDelegate().supportsInactiveAccounts();
    }

    @Override
    public boolean supportsNestedGroups() {
        return getDelegate().supportsNestedGroups();
    }

    @Override
    public boolean supportsPasswordExpiration() {
        return getDelegate().supportsPasswordExpiration();
    }

    @Override
    public boolean supportsSettingEncryptedCredential() {
        return getDelegate().supportsSettingEncryptedCredential();
    }

    @Override
    public boolean isRolesDisabled() {
        return getDelegate().isRolesDisabled();
    }

    @Override
    @Nonnull
    public Iterable<Membership> getMemberships() throws OperationFailedException {
        return getDelegate().getMemberships();
    }

    @Override
    @Nonnull
    public RemoteDirectory getAuthoritativeDirectory() {
        return getDelegate().getAuthoritativeDirectory();
    }

    @Override
    public void expireAllPasswords() throws OperationFailedException {
        getDelegate().expireAllPasswords();
    }

    @Override
    @Nullable
    public Set<String> getValues(String key) {
        return getDelegate().getValues(key);
    }

    @Override
    public String getValue(String key) {
        return getDelegate().getValue(key);
    }

    @Override
    public Set<String> getKeys() {
        return getDelegate().getKeys();
    }

    @Override
    public boolean isEmpty() {
        return getDelegate().isEmpty();
    }

    @Override
    public AvatarReference getUserAvatarByName(String username, int sizeHint)
            throws UserNotFoundException, OperationFailedException {
        return getDelegate().getUserAvatarByName(username, sizeHint);
    }

    /**
     * @return the directory to delegate method calls to
     */
    protected abstract RemoteDirectory getDelegate();

}
