package com.atlassian.crowd.search.ldap;

import com.atlassian.crowd.directory.MicrosoftActiveDirectory;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.directory.ldap.mapper.attribute.ObjectGUIDMapper;
import com.atlassian.crowd.directory.ldap.mapper.attribute.UserAccountControlMapper;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.ldap.filter.EqualsExternalIdFilter;
import com.atlassian.crowd.search.query.entity.restriction.PropertyRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;

import org.springframework.ldap.filter.Filter;
import org.springframework.ldap.filter.NotFilter;

/**
 * An specialisation of {@link LDAPQueryTranslaterImpl} that can translate Active Directory
 * <a href="http://support.microsoft.com/kb/269181">enabled/disabled user queries</a>.
 *
 * @since v2.7
 */
public class ActiveDirectoryQueryTranslaterImpl extends LDAPQueryTranslaterImpl {
    private static final int UF_ACCOUNTDISABLE_MASK = MicrosoftActiveDirectory.UF_ACCOUNTDISABLE;

    @Override
    protected Filter booleanTermRestrictionAsFilter(EntityDescriptor entityDescriptor,
                                                    PropertyRestriction<Boolean> termRestriction,
                                                    LDAPPropertiesMapper ldapPropertiesMapper) {
        if (entityDescriptor.getEntityType() == Entity.USER && termRestriction.getProperty().equals(UserTermKeys.ACTIVE)) {
            if (termRestriction.getValue()) {
                // querying for enabled users
                return new NotFilter(BitwiseFilter.or(UserAccountControlMapper.ATTRIBUTE_KEY, UF_ACCOUNTDISABLE_MASK));
            } else {
                // querying for disabled users
                return BitwiseFilter.and(UserAccountControlMapper.ATTRIBUTE_KEY, UF_ACCOUNTDISABLE_MASK);
            }
        } else if (entityDescriptor.getEntityType() == Entity.GROUP && termRestriction.getProperty().equals(GroupTermKeys.ACTIVE)) {
            if (termRestriction.getValue()) {
                // groups are always active = true, so no need to add a restriction
                return new EverythingResult();
            } else {
                // no group is ever active = false, so no results
                return new NothingResult();
            }
        } else {
            // if boolean term restrictions are for anything other than the group/user active flag, then throw exception
            throw new IllegalArgumentException("Boolean restrictions for property " + termRestriction.getProperty().getPropertyName() + " are not supported");
        }
    }

    /**
     * Returns a filter that applies byte String encoding to value when creating filter for ExternalId property.
     * When searching by ObjectGUID the ActiveDirectory expects the value to be in 'Byte String' form, without any additional encoding.
     * In this case it's required to pass the PropertyRestriction containing String value of ObjectGUID.
     *
     * For every other property it returns a filter with regular encoding (delegates to superclass).
     *
     * @param propertyName    attribute name to be queried in AD
     * @param termRestriction PropertyRestriction containing the value that will be searched for
     * @return filter to be used in AD search
     * @see com.atlassian.crowd.directory.ldap.util.GuidHelper
     */
    @Override
    protected Filter getStringTermEqualityFilter(final String propertyName, final PropertyRestriction<String> termRestriction) {
        // attribute name is case-insensitive during filtering - https://social.technet.microsoft.com/Forums/sharepoint/en-US/39305ac4-b855-4113-80ee-be7324e4fa97/is-ldap-attribute-are-case-sensitive-?forum=winserverDS
        if (ObjectGUIDMapper.ATTRIBUTE_KEY.equalsIgnoreCase(propertyName)) {
            return new EqualsExternalIdFilter(propertyName, termRestriction.getValue());
        } else {
            return super.getStringTermEqualityFilter(propertyName, termRestriction);
        }
    }
}
