package com.atlassian.crowd.model.group;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.atlassian.crowd.model.LDAPDirectoryEntity;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

public class LDAPGroupWithAttributes implements GroupWithAttributes, LDAPDirectoryEntity {
    private final String dn;
    private final String name;
    private final Long directoryId;
    private final GroupType type;
    private final boolean active;
    private final String description;
    private final String externalId;

    private final Map<String, Set<String>> attributes = new HashMap<String, Set<String>>();

    public LDAPGroupWithAttributes(String dn, GroupTemplateWithAttributes group) {
        Validate.isTrue(StringUtils.isNotBlank(dn));
        Validate.notNull(group, "group template cannot be null");
        Validate.notNull(group.getDirectoryId(), "directoryId cannot be null");
        Validate.notNull(group.getName(), "group name cannot be null");
        Validate.notNull(group.getType(), "group type cannot be null");

        this.dn = dn;
        this.directoryId = group.getDirectoryId();
        this.name = group.getName();
        this.active = group.isActive();
        this.type = group.getType();
        this.description = group.getDescription();
        this.externalId = group.getExternalId();

        // clone the attributes map
        for (Map.Entry<String, Set<String>> entry : group.getAttributes().entrySet()) {
            attributes.put(entry.getKey(), new HashSet<String>(entry.getValue()));
        }
    }

    @Override
    public String getDn() {
        return dn;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    @Override
    public GroupType getType() {
        return type;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Set<String> getValues(String attributeName) {
        if (attributes.containsKey(attributeName)) {
            return Collections.unmodifiableSet(attributes.get(attributeName));
        } else {
            return Collections.emptySet();
        }
    }

    @Override
    public String getValue(String attributeName) {
        Set<String> values = getValues(attributeName);
        if (values != null && !values.isEmpty()) {
            return values.iterator().next();
        } else {
            return null;
        }
    }

    @Override
    public Set<String> getKeys() {
        return Collections.unmodifiableSet(attributes.keySet());
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    @Nullable
    public String getExternalId() {
        return externalId;
    }

    @Override
    public boolean equals(final Object o) {
        return GroupComparator.equalsObject(this, o);
    }

    @Override
    public int hashCode() {
        return GroupComparator.hashCode(this);
    }

    @Override
    public int compareTo(Group group) {
        return GroupComparator.compareTo(this, group);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("dn", dn).
                append("name", name).
                append("directoryId", directoryId).
                append("type", type).
                append("active", active).
                append("description", description).
                append("externalId", externalId).
                append("attributes", attributes).
                toString();
    }
}
