package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.DeduplicatingDnMapperDecorator;
import com.atlassian.crowd.directory.MicrosoftActiveDirectory;
import com.atlassian.crowd.directory.ldap.cache.UsnChangedCacheRefresherIncSyncException;
import com.atlassian.crowd.directory.ldap.mapper.attribute.user.MemberOfOverlayMapper;
import com.atlassian.crowd.directory.rfc4519.RFC4519DirectoryMembershipsIterableBuilder;
import com.atlassian.crowd.directory.synchronisation.CacheSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.PartialSynchronisationResult;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.DirectoryEntities;
import com.atlassian.crowd.model.Tombstone;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.LDAPGroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.LDAPUserWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.util.concurrent.ThreadFactories;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.primitives.Longs;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.atlassian.crowd.directory.RFC4519Directory.DN_MAPPER;
import static com.google.common.collect.Iterables.transform;

/**
 * Retrieves latest changes from MS Active Directory in order to allow "delta" cache refreshes.
 * <p>
 * See http://msdn.microsoft.com/en-us/library/ms677625.aspx and http://msdn.microsoft.com/en-us/library/ms677627%28VS.85%29.aspx
 * for details on polling Microsoft Active Directory.
 * <p>
 * This class is guaranteed to be run from a single thread at a time by per directory.  This means it does not need to
 * worry about race-conditions, but still must consider safe publication of variables (per directory).
 *
 * The incremental sync of users comes in two flavors:
 * <ul>
 * <li>Legacy mode - based on usnChanged attribute only
 * <li>New 'bulletproof' mode (default) - using both usnChanged and ObjectGUID diff between internal and remote directory
 * </ul>
 * It is possible to switch back to 'legacy mode' by supplying a system property named {@link #PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC} set to {@code true}.
 */
public class UsnChangedCacheRefresher extends AbstractCacheRefresher<LDAPGroupWithAttributes> implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(UsnChangedCacheRefresher.class);
    public static final String PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC = "crowd.use.legacy.ad.incremental.sync";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private final MicrosoftActiveDirectory activeDirectory;

    final LDAPEntityNameMap<LDAPUserWithAttributes> userMap = new LDAPEntityNameMap<>();
    private final LDAPEntityNameMap<LDAPGroupWithAttributes> groupMap = new LDAPEntityNameMap<>();

    private Future<List<LDAPUserWithAttributes>> userListFuture;
    private Future<List<LDAPGroupWithAttributes>> groupListFuture;
    private Set<String> groupsDnsToUpdate = new HashSet<>(); // Stores dns of groups of new users found during incremental synchronisation, cleared before and after every sync
    private Set<String> primaryGroupSids = new HashSet<>(); // Stores sids of primary groups of new users found during incremental synchronisation, cleared before and after every sync

    private final boolean useLegacyADIncrementalSync = Boolean.valueOf(System.getProperty(PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC, "false"));

    public UsnChangedCacheRefresher(final MicrosoftActiveDirectory activeDirectory) {
        super(activeDirectory);
        this.activeDirectory = activeDirectory;
    }

    @Override
    public CacheSynchronisationResult synchroniseChanges(final DirectoryCache directoryCache, @Nullable String highestCommittedUsn) throws OperationFailedException {
        if (!isIncrementalSyncEnabled()) {
            return CacheSynchronisationResult.FAILURE;
        }

        final ActiveDirectoryTokenHolder maybeTokenHolder = deserializeDirectorySyncToken(highestCommittedUsn);
        // We don't have highestCommittedUsn, fallback to full
        if (maybeTokenHolder == null) {
            log.info("Synchronisation token not present, full sync of directory [{}] is necessary before incremental sync is possible.",
                    activeDirectory.getDirectoryId());
            return CacheSynchronisationResult.FAILURE;
        }
        final String currentInvocationId = activeDirectory.fetchInvocationId();
        if (!Objects.equals(currentInvocationId, maybeTokenHolder.getInvocationId())) {
            log.info("Last incremental synchronization took place for AD instance with invocation id '{}', " +
                            "current instance has invocation id of '{}', falling back to full",
                    maybeTokenHolder.getInvocationId(), currentInvocationId);
            return CacheSynchronisationResult.FAILURE;
        }


        final long lastUsnParsed = maybeTokenHolder.getLastUsnChanged();
        // Find the latest USN-Changed value for the AD server
        final long currentHighestCommittedUSN = activeDirectory.fetchHighestCommittedUSN();

        synchroniseUserChanges(directoryCache, lastUsnParsed);
        synchroniseGroupChanges(directoryCache, lastUsnParsed);

        return new CacheSynchronisationResult(true, serializeDirectorySyncToken(currentInvocationId, currentHighestCommittedUSN));
    }

    public CacheSynchronisationResult synchroniseAll(final DirectoryCache directoryCache) throws OperationFailedException {
        final ExecutorService queryExecutor = Executors.newFixedThreadPool(3, ThreadFactories.namedThreadFactory("CrowdUsnChangedCacheRefresher"));
        try {
            userListFuture = queryExecutor.submit(() -> {
                final long start = System.currentTimeMillis();
                log.debug("loading remote users");
                final List<LDAPUserWithAttributes> ldapUsers;
                if (isUserAttributeSynchronisationEnabled()) {
                    //noinspection unchecked
                    ldapUsers = (List) activeDirectory.searchUsers(QueryBuilder
                            .queryFor(UserWithAttributes.class, EntityDescriptor.user())
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                } else {
                    //noinspection unchecked
                    ldapUsers = (List) activeDirectory.searchUsers(QueryBuilder
                            .queryFor(User.class, EntityDescriptor.user())
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                }
                log.info("found [ {} ] remote users in [ {}ms ]", ldapUsers.size(), (System.currentTimeMillis() - start));
                return ldapUsers;
            });
            groupListFuture = queryExecutor.submit(() -> {
                final long start = System.currentTimeMillis();
                log.debug("loading remote groups");
                final List<LDAPGroupWithAttributes> ldapGroups;
                if (isGroupAttributeSynchronisationEnabled()) {
                    //noinspection unchecked
                    ldapGroups = (List) activeDirectory.searchGroups(QueryBuilder
                            .queryFor(GroupWithAttributes.class, EntityDescriptor.group(GroupType.GROUP))
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                } else {
                    //noinspection unchecked
                    ldapGroups = (List) activeDirectory.searchGroups(QueryBuilder
                            .queryFor(Group.class, EntityDescriptor.group(GroupType.GROUP))
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                }
                log.info("found [ " + ldapGroups.size() + " ] remote groups in [ " + (System.currentTimeMillis() - start) + "ms ]");
                return ldapGroups;
            });
            // Do standard synchroniseAll
            super.synchroniseAll(directoryCache);

            if (!isIncrementalSyncEnabled()) {
                return new CacheSynchronisationResult(true, null);
            }

            return new CacheSynchronisationResult(true, serializeDirectorySyncToken(
                    activeDirectory.fetchInvocationId(),
                    activeDirectory.fetchHighestCommittedUSN()
            ));
        } finally {
            queryExecutor.shutdown();

            userListFuture = null;
            groupListFuture = null;
        }
    }

    @Override
    protected PartialSynchronisationResult<? extends UserWithAttributes> synchroniseAllUsers(final DirectoryCache directoryCache) throws OperationFailedException {
        final Date syncStartDate = new Date();

        try {
            final List<LDAPUserWithAttributes> ldapUsers = userListFuture.get();

            userMap.putAll(ldapUsers);
            //the order of this operations is relevant for rename to work properly
            directoryCache.deleteCachedUsersNotIn(ldapUsers, syncStartDate);
            directoryCache.addOrUpdateCachedUsers(ldapUsers, syncStartDate);

            return new PartialSynchronisationResult<>(ldapUsers);
        } catch (final InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new OperationFailedException("background query interrupted", e);
        } catch (final ExecutionException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    protected PartialSynchronisationResult<LDAPGroupWithAttributes> synchroniseAllGroups(DirectoryCache directoryCache) throws OperationFailedException {
        final Date syncStartDate = new Date();

        try {
            List<LDAPGroupWithAttributes> ldapGroups = groupListFuture.get();

            //CWD-2504: We filter out the duplicate groups to stop the runtime exception that will occur during membership
            // synchronization. There is no point registering a group we are not going to use.
            ldapGroups = Collections.unmodifiableList(DirectoryEntities.filterOutDuplicates(ldapGroups));

            groupMap.putAll(ldapGroups);

            directoryCache.deleteCachedGroupsNotIn(GroupType.GROUP, ldapGroups, syncStartDate);
            directoryCache.addOrUpdateCachedGroups(ldapGroups, syncStartDate);
            return new PartialSynchronisationResult<>(ldapGroups);
        } catch (final InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new OperationFailedException("background query interrupted", e);
        } catch (final ExecutionException e) {
            throw new OperationFailedException(e);
        }
    }

    private void synchroniseUserChangesUsn(final DirectoryCache directoryCache, long highestCommittedUsn) throws OperationFailedException {
        long start = System.currentTimeMillis();
        log.debug("loading changed remote users");
        final List<LDAPUserWithAttributes> updatedUsers = activeDirectory.findAddedOrUpdatedUsersSince(highestCommittedUsn);
        final List<Tombstone> tombstones = activeDirectory.findUserTombstonesSince(highestCommittedUsn);
        userMap.putAll(updatedUsers);

        log.info("found [ {} ] changed, [ {} ] deleted remote users in [ {}ms ]", updatedUsers.size(), tombstones.size(), (System.currentTimeMillis() - start));

        // calculate removed principals
        final Set<String> tombstonesGuids = ImmutableSet.copyOf(transform(tombstones, Tombstone::getObjectGUID));

        directoryCache.deleteCachedUsersByGuid(tombstonesGuids);
        userMap.removeAllByGuid(tombstonesGuids);

        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedUsers(updatedUsers, null);
    }

    /**
     * Returns user guids to be added and removed unless it detects the incremental sync is not possible in which case it throws an UsnChangedCacheRefresherIncSyncException.
     * This method intentionally does two things - it tries to fail as fast as possible. Some operations, like AD query, might be time consuming and we don't want to
     * wait unnecessarily long only to find out the incremental sync cannot be performed.
     *
     * @param directoryCache directory cache
     * @return Pair containing user guids to be added (left) and guids to be removed (right) from cache
     * @throws OperationFailedException                 when directoryCache or external directory operations throws OperationFailedException
     * @throws UsnChangedCacheRefresherIncSyncException when it detects the incremental sync should not be performed due to data inconsistency
     */
    private Pair<? extends Set<String>, ? extends Set<String>> validateAndReturnUserGuidsToAddAndDelete(final DirectoryCache directoryCache) throws OperationFailedException {
        if (!activeDirectory.isUsersExternalIdConfigured()) {
            throw new UsnChangedCacheRefresherIncSyncException("externalId attribute is not configured in directory.");
        }
        log.debug("loading changed users");

        // check cache first, in case any null/empty guids are found it will fail faster
        final Set<String> userGuidsInCache = ImmutableSet.copyOf(directoryCache.getAllUserGuids());
        if (userGuidsInCache.size() != directoryCache.getUserCount()) {
            throw new UsnChangedCacheRefresherIncSyncException("Cache returned different number of guids and users (possible reason is overlapping guids in cache," +
                    " most likely null/empty values).");
        }
        if (userGuidsInCache.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty user guids returned from cache. Falling back to a full sync in order to populate the guids");
        }

        // fetch all user guids from AD
        final Set<String> userGuidsInAd = activeDirectory.findAllUserGuids();
        log.info("Found [ {} ] user GUIDs in cache, [ {} ] user GUIDs in remote directory", userGuidsInCache.size(), userGuidsInAd.size());
        // in case the externalId directory attribute is misconfigured or the AD itself is misconfigured we might get empty guids
        if (userGuidsInAd.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty user guids returned from AD. Possible reasons are externalId attribute value in directory configuration" +
                    " or AD server configuration.");
        }

        return Pair.of(Sets.difference(userGuidsInAd, userGuidsInCache), Sets.difference(userGuidsInCache, userGuidsInAd));
    }

    /**
     * This synchronizes users based on guids and usn.
     * Users to be added/removed are determined using diff in guids from internal and remote directory.
     * Users to be updated are calculated using usn.
     *
     * @param directoryCache      directory cache
     * @param highestCommittedUsn synchronisation token
     * @throws OperationFailedException                 when directoryCache or external directory operations throws OperationFailedException
     * @throws UsnChangedCacheRefresherIncSyncException when incremental sync failed or cannot be performed due to data consistency problems
     */
    private void synchroniseUserChangesGuid(final DirectoryCache directoryCache, final Long highestCommittedUsn) throws OperationFailedException {
        final long start = System.currentTimeMillis();

        final Pair<? extends Set<String>, ? extends Set<String>> guidsToAddAndRemove = validateAndReturnUserGuidsToAddAndDelete(directoryCache);
        final Set<String> guidsToAdd = guidsToAddAndRemove.getLeft();
        final Set<String> guidsToRemove = guidsToAddAndRemove.getRight();

        // when queried for changes by usn the AD will return both updated and newly added users
        // we want to perform updates and additions separately in order to avoid problems (existing user X renamed to Y, new user X added)
        // therefore we filter this result
        final ImmutableMap.Builder<String, LDAPUserWithAttributes> usersToAddByGuidBuilder = ImmutableMap.builder();
        final ImmutableList.Builder<LDAPUserWithAttributes> usersToUpdateBuilder = ImmutableList.builder();
        for (LDAPUserWithAttributes user : activeDirectory.findAddedOrUpdatedUsersSince(highestCommittedUsn)) {
            final String externalId = user.getExternalId();
            // in the unlikely event of AD returning null or empty externalIds right after we checked this with the validateAndReturnUserGuidsToAddAndDelete()
            // we throw an exception to fallback to a full sync
            if (StringUtils.isEmpty(externalId)) {
                throw new UsnChangedCacheRefresherIncSyncException("A null or empty guid retrieved from AD.");
            }
            if (guidsToAdd.contains(externalId)) {
                // this user is already referenced by toAdd set so it needs to be filtered out.
                // in order to reduce the number of calls to AD, when we fetch users to be added from AD by guid, we store this user in cache.
                usersToAddByGuidBuilder.put(externalId, user);
            } else {
                usersToUpdateBuilder.add(user);
            }
        }
        final ImmutableMap<String, LDAPUserWithAttributes> usersToAddByGuid = usersToAddByGuidBuilder.build();
        final ImmutableList<LDAPUserWithAttributes> usersToUpdate = usersToUpdateBuilder.build();

        log.info("scanned and compared [ {} ] users to delete, [ {} ] users to add, [ {} ] users to update in DB cache in [ {}ms ]",
                guidsToRemove.size(), guidsToAdd.size(), usersToUpdate.size(), (System.currentTimeMillis() - start));
        logUserChanges(guidsToRemove, guidsToAdd, usersToAddByGuid, usersToUpdate);

        // 1 - delete users by guid
        directoryCache.deleteCachedUsersByGuid(guidsToRemove);
        userMap.removeAllByGuid(guidsToRemove);

        // 2 - update changed users (usn)
        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedUsers(usersToUpdate, null);
        userMap.putAll(usersToUpdate);

        Function<String, LDAPUserWithAttributes> fetchUserByGuid = externalId -> {
            try {
                final LDAPUserWithAttributes userFromCache = usersToAddByGuid.get(externalId);
                if (userFromCache != null) {
                    return userFromCache;
                } else {
                    log.trace("Detected additional user with external id '{}' (probably added during the sync). Fetching it from AD...", externalId);
                    return activeDirectory.findUserByExternalId(externalId);
                }
            } catch (UserNotFoundException | OperationFailedException e) {
                log.warn("Failed to fetch user by objectGUID '{}' from ActiveDirectory", externalId, e);
            }
            throw new UsnChangedCacheRefresherIncSyncException("Problems while looking up users by objectGUID in ActiveDirectory detected, falling back to a full sync.");
        };

        // 3 - add new users by guid, but first fetch their User data
        final List<LDAPUserWithAttributes> newUsers = guidsToAdd.stream()
                .map(fetchUserByGuid)
                .collect(Collectors.toList());
        directoryCache.addOrUpdateCachedUsers(newUsers, null);
        userMap.putAll(newUsers);

        for (final LDAPUserWithAttributes newUser : newUsers) {
            final Set<String> groups = newUser.getValues(MemberOfOverlayMapper.ATTRIBUTE_KEY);
            if (groups != null && groups.size() > 0) {
                groupsDnsToUpdate.addAll(groups);
            }
            if (activeDirectory.getLdapPropertiesMapper().isPrimaryGroupSupported()) {
                final Optional<String> primaryGroupSidOrEmpty = activeDirectory.getPrimaryGroupSIDOfUser(newUser);
                primaryGroupSidOrEmpty.ifPresent(s -> primaryGroupSids.add(s));
            }
        }
    }

    void synchroniseUserChanges(final DirectoryCache directoryCache, final Long highestCommittedUsn) throws OperationFailedException {
        if (useLegacyADIncrementalSync) {
            synchroniseUserChangesUsn(directoryCache, highestCommittedUsn);
        } else {
            synchroniseUserChangesGuid(directoryCache, highestCommittedUsn);
        }
    }

    private void synchroniseGroupChanges(DirectoryCache directoryCache, long highestCommittedUsn) throws OperationFailedException {
        final long start = System.currentTimeMillis();
        log.debug("loading changed groups");

        final Set<String> groupGuidsInCache = getAndValidateGroupGuidsFromCache(directoryCache);
        final Set<String> groupGuidsInAd = getSynchronizableGroupGuidsFromAd(directoryCache);
        final Set<String> groupGuidsToRemove = Sets.difference(groupGuidsInCache, groupGuidsInAd);
        log.info("Found [ {} ] group GUIDs in cache, [ {} ] group GUIDs in remote directory", groupGuidsInCache.size(), groupGuidsInAd.size());

        // when queried for changes by usn the AD will return both updated and newly added groups
        // we want to perform updates and additions separately in order to avoid problems (existing user X renamed to Y, new user X added)
        // therefore we filter this result
        Pair<List<LDAPGroupWithAttributes>, List<LDAPGroupWithAttributes>> addedAndUpdated =
                findAndValidateAddedAndUpdatedGroupsSince(highestCommittedUsn, groupGuidsInCache, groupGuidsInAd);

        log.info("scanned and compared [ {} ] groups to delete, [ {} ] groups to add, [ {} ] groups to update in DB cache in [ {}ms ]"
                , groupGuidsToRemove.size(), addedAndUpdated.getLeft().size(), addedAndUpdated.getRight().size(), (System.currentTimeMillis() - start));

        syncGroups(directoryCache, groupGuidsToRemove, addedAndUpdated.getLeft(), addedAndUpdated.getRight());
    }

    private Pair<List<LDAPGroupWithAttributes>, List<LDAPGroupWithAttributes>> findAndValidateAddedAndUpdatedGroupsSince(
            long highestCommittedUsn, Set<String> groupGuidsInCache, Set<String> groupGuidsInAd) throws OperationFailedException {
        final Map<String, LDAPGroupWithAttributes> newGroups = new HashMap<>();
        final ImmutableList.Builder<LDAPGroupWithAttributes> groupsToUpdateBuilder = ImmutableList.builder();
        for (LDAPGroupWithAttributes group : activeDirectory.findAddedOrUpdatedGroupsSince(highestCommittedUsn)) {
            final String externalId = group.getExternalId();
            // in the unlikely event of AD returning null or empty externalIds right after we checked this with the validateAndReturnUserGuidsToAddAndDelete()
            // we throw an exception to fallback to a full sync
            if (StringUtils.isEmpty(externalId)) {
                throw new UsnChangedCacheRefresherIncSyncException("A null or empty guid retrieved from AD.");
            }
            if (groupGuidsInCache.contains(externalId)) {
                groupsToUpdateBuilder.add(group);
            } else if (groupGuidsInAd.contains(externalId)) {
                // this group is already referenced by toAdd set so it needs to be filtered out.
                // in order to reduce the number of calls to AD, when we fetch users to be added from AD by guid, we store this user in cache.
                newGroups.put(externalId, group);
            }
            // if group does not appear in groupGuidsInAd it means that it is not synchronizable, either duplicated,
            // or there is a local group with the same name
        }
        Set<String> missingGuids = Sets.difference(Sets.difference(groupGuidsInAd, groupGuidsInCache), newGroups.keySet());
        if (!missingGuids.isEmpty()) {
            log.warn("Failed to fetch groups by objectGUIDs '{}' from ActiveDirectory", missingGuids);
            throw new UsnChangedCacheRefresherIncSyncException("Problems while looking up groups by objectGUID in ActiveDirectory detected, falling back to a full sync.");
        }
        return Pair.of(ImmutableList.copyOf(newGroups.values()), groupsToUpdateBuilder.build());
    }

    private Set<String> getAndValidateGroupGuidsFromCache(DirectoryCache directoryCache) throws OperationFailedException {
        // check cache first, in case any null/empty guids are found it will fail faster
        final Set<String> groupGuidsInCache = ImmutableSet.copyOf(directoryCache.getAllGroupGuids());
        if (groupGuidsInCache.size() != directoryCache.getExternalCachedGroupCount()) {
            throw new UsnChangedCacheRefresherIncSyncException("Cache returned different number of guids and non-local groups (possible reason is overlapping guids in cache," +
                    " most likely null/empty values).");
        }
        if (groupGuidsInCache.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty group guids returned from cache. Falling back to a full sync in order to populate the guids");
        }
        return groupGuidsInCache;
    }

    private Set<String> getSynchronizableGroupGuidsFromAd(DirectoryCache directoryCache) throws OperationFailedException {
        if (!activeDirectory.isGroupExternalIdConfigured()) {
            throw new UsnChangedCacheRefresherIncSyncException("externalId attribute is not configured in directory.");
        }
        Collection<Pair<String, String>> namesAndGuids =
                DirectoryEntities.filterOutDuplicates(activeDirectory.findAllGroupNamesAndGuids(), Pair::getLeft);
        Predicate<String> isLocalGroup = IdentifierUtils.containsIdentifierPredicate(directoryCache.getAllLocalGroupNames());
        final Set<String> groupGuidsInAd = namesAndGuids.stream()
                .filter(group -> !isLocalGroup.test(group.getLeft()))
                .map(Pair::getRight)
                .collect(Collectors.toSet());
        // in case the externalId directory attribute is misconfigured or the AD itself is misconfigured we might get empty guids
        if (groupGuidsInAd.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty groups guids returned from AD. Possible reasons are externalId attribute value in directory configuration" +
                    " or AD server configuration.");
        }
        return groupGuidsInAd;
    }

    private void syncGroups(DirectoryCache directoryCache, Set<String> groupGuidsToRemove,
                    Collection<LDAPGroupWithAttributes> newGroups, Collection<LDAPGroupWithAttributes> groupsToUpdate)
            throws OperationFailedException {
        // 1 - delete groups by guid
        directoryCache.deleteCachedGroupsByGuids(groupGuidsToRemove);
        groupMap.removeAllByGuid(groupGuidsToRemove);

        // 2 - update changed groups (usn)
        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedGroups(groupsToUpdate, null);
        groupMap.putAll(groupsToUpdate);

        directoryCache.addOrUpdateCachedGroups(newGroups, null);
        groupMap.putAll(newGroups);

        //First synchronise changed and new groups
        synchroniseMemberships(
                Sets.union(
                        ImmutableSet.copyOf(newGroups),
                        ImmutableSet.copyOf(groupsToUpdate)),
                directoryCache, false);

        //After that synchronise groups by Dns and Sids (after new groups, because we may have nested group memberships)
        synchroniseMemberships(
                Sets.union(
                        ImmutableSet.copyOf(activeDirectory.searchGroupsByDns(groupsDnsToUpdate)),
                        ImmutableSet.copyOf(activeDirectory.searchGroupsBySids(primaryGroupSids))),
                directoryCache, false);
    }

    @Override
    protected Iterable<Membership> getMemberships(final Collection<LDAPGroupWithAttributes> groupsToInclude, boolean isFullSync) throws OperationFailedException {
        try {
            final Map<LdapName, String> users = userMap.toLdapNameKeyedMap();
            final Map<LdapName, String> groups = groupMap.toLdapNameKeyedMap();
            final RFC4519DirectoryMembershipsIterableBuilder iterableBuilder = new RFC4519DirectoryMembershipsIterableBuilder()
                    .forConnector(activeDirectory)
                    .forGroups(groupsToInclude)
                    .withDnMapper(new DeduplicatingDnMapperDecorator(DN_MAPPER));
            if (isFullSync) {
                iterableBuilder.withFullCache(users, groups);
            } else {
                iterableBuilder.withPartialCache(users, groups);
            }
            return iterableBuilder.build();
        } catch (final InvalidNameException e) {
            throw new OperationFailedException("Failed to get directory memberships due to invalid DN", e);
        }
    }

    private ActiveDirectoryTokenHolder deserializeDirectorySyncToken(String token) {
        if (!Strings.isNullOrEmpty(token)) {
            try {
                return OBJECT_MAPPER.readValue(token, ActiveDirectoryTokenHolder.class);
            } catch (IOException e) {
                if (Longs.tryParse(token) != null) {
                    log.warn("Cannot proceed with incremental synchronization as sync token '{}' does not contain the invocation id, falling back to FULL", token);
                } else {
                    log.warn("Cannot parse sync token '{}', falling back to FULL", token);
                }
            }
        }
        return null;
    }

    private String serializeDirectorySyncToken(final String currentInvocationId,
                                               final long currentHighestCommittedUsn) {
        try {
            return OBJECT_MAPPER.writeValueAsString(new ActiveDirectoryTokenHolder(currentInvocationId, currentHighestCommittedUsn));
        } catch (IOException e) {
            log.warn("Cannot serialize synchronisation token obtained from Azure AD. Last invocation id: '{}', highestCommittedUsn: '{}'",
                    currentInvocationId,
                    currentHighestCommittedUsn, e);
            return null;
        }
    }

    private void logUserChanges(Set<String> guidsToRemove, Set<String> guidsToAdd, ImmutableMap<String, LDAPUserWithAttributes> usersToAddByGuid, ImmutableList<LDAPUserWithAttributes> usersToUpdate) {
        if (log.isTraceEnabled()) {
            final String DELIMITER = ",\n";
            log.trace("Users to remove: {}", String.join(DELIMITER, guidsToRemove));
            log.trace("Users to add: {}", guidsToAdd.stream()
                    .map(guid -> Optional.ofNullable(usersToAddByGuid.get(guid))
                        .map(user -> String.format("%s(%s)", guid, user.getName()))
                        .orElse(String.format("%s", guid)))
                    .collect(Collectors.joining(DELIMITER)));
            log.trace("Users to update: {}", usersToUpdate.stream()
                    .map(user -> String.format("%s(%s)", user.getExternalId(), user.getName()))
                    .collect(Collectors.joining(DELIMITER)));
        }
    }
}
