package com.atlassian.crowd.directory.rfc4519;

import com.atlassian.crowd.directory.RFC4519Directory;
import com.atlassian.crowd.directory.ldap.mapper.ContextMapperWithRequiredAttributes;
import com.atlassian.crowd.model.LDAPDirectoryEntity;
import com.atlassian.crowd.model.group.LDAPGroupWithAttributes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.naming.ldap.LdapName;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import static com.atlassian.crowd.directory.RFC4519Directory.DN_MAPPER;

public class RFC4519DirectoryMembershipsIterableBuilder {
    public static final int PARTITION_SIZE = Integer.getInteger("com.atlassian.crowd.directory.RFC4519DirectoryMembershipsIterable.PARTITION_SIZE", 1000);
    static final String FORCE_LOOKUP_MISSING_NAMES_PROPERTY = "com.atlassian.crowd.directory.RFC4519DirectoryMembershipsIterable.FORCE_LOOKUP_MISSING_NAMES";
    private static final boolean FORCE_LOOKUP_MISSING_NAMES = Boolean.getBoolean(FORCE_LOOKUP_MISSING_NAMES_PROPERTY);
    private static final Logger log = LoggerFactory.getLogger(RFC4519DirectoryMembershipsIterableBuilder.class);

    private RFC4519Directory connector;
    private Map<LdapName, String> users;
    private Map<LdapName, String> groups;
    private boolean fullCache;
    private Map<LdapName, String> groupsToInclude;
    private Integer membershipBatchSize;
    private ContextMapperWithRequiredAttributes<LdapName> dnMapper = DN_MAPPER;

    public RFC4519DirectoryMembershipsIterable build() {
        final Map<LdapName, String> groupsToInclude = this.groupsToInclude == null ? groups : this.groupsToInclude;
        int batchSize = membershipBatchSize == null ? PARTITION_SIZE : membershipBatchSize;
        if (fullCache) {
            return new RFC4519DirectoryMembershipsIterableWithFullCache(connector, users, groups, groupsToInclude, batchSize, dnMapper);
        } else {
            return new RFC4519DirectoryMembershipsIterable(connector, users, groups, groupsToInclude, batchSize, dnMapper);
        }
    }

    public RFC4519DirectoryMembershipsIterableBuilder withDnMapper(ContextMapperWithRequiredAttributes<LdapName> dnMapper) {
        this.dnMapper = dnMapper;
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder forConnector(RFC4519Directory connector) {
        this.connector = connector;
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder withFullCache(Map<LdapName, String> users, Map<LdapName, String> groups) {
        if (FORCE_LOOKUP_MISSING_NAMES) {
            logForceLookupEnabled();
        } else {
            fullCache = true;
        }
        this.users = users;
        this.groups = groups;
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder withPartialCache(Map<LdapName, String> users, Map<LdapName, String> groups) {
        fullCache = false;
        this.users = users;
        this.groups = groups;
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder forGroups(Map<LdapName, String> groupsToInclude) {
        this.groupsToInclude = groupsToInclude;
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder forGroups(Collection<LDAPGroupWithAttributes> groupsToInclude) {
        this.groupsToInclude = groupsToInclude.stream().collect(Collectors.toMap(LDAPDirectoryEntity::getLdapName, LDAPGroupWithAttributes::getName));
        return this;
    }

    public RFC4519DirectoryMembershipsIterableBuilder withCustomBatchSize(int membershipBatchSize) {
        this.membershipBatchSize = membershipBatchSize;
        return this;
    }

    private static void logForceLookupEnabled() {
        log.debug("Returning a RFC4519 memberships iterable with forced lookups as the {} property is enabled", FORCE_LOOKUP_MISSING_NAMES_PROPERTY);
    }
}
