package com.atlassian.crowd.directory.ldap.util;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * <p>This class is used to normalise DNs so that equivalent DNs will be transformed into equal strings.
 * Often, a better way to compare DNs is to use {@link javax.naming.ldap.LdapName}.</p>
 * <p>Although this class is not marked as deprecated, as there is no direct replacement, you probably don't want
 * to be making new use of it.</p>
 */
public class DNStandardiser {
    /**
     * Converts a DN string into a "standard"
     * DN string representation consumable by
     * caches.
     *
     * This is particularly useful when matching
     * group memberDNs to entity DNs as the two
     * DN formats need to be equivalent.
     *
     * AD returns DN's that have mixed case attribute labels, eg. member attribute might look like: CN=user-1100,CN=Users,DC=sydney,DC=atlassian,DC=com for AD.
     *
     * Since everything goes through a dirContextAdapter (thanks for coming in: SpringLDAP), when we call context.getDn()
     * in the UserContextMapper/GroupContextMapper, the DN we get back is a SpringLDAP parsed DN (ie. forces lower-casing of
     * attribute names). However, when we get back memberDNs for a group, these attributes values are pure String representaions
     * of the DN returned from the server. This will cause the memberDN to not match the DN of any object previously seen as
     * the memberDN will not be forced toLowerCase or forced to compaction (",") or comma-space-delimmition (", ") for RDN
     * components of the memberDN.
     *
     * It is critical that the DN and memberDN are comparable based on a raw String comparison (String.equals)
     * for things like group membership determination and cache consistency.
     *
     * Therefore, if it is known that
     * the directory server always returns DNs
     * that do not have ", " but have "," separating
     * the naming components, then we do not need
     * to force "proper" (ie. complete + expensive)
     * standardisation. Thus if <code>forceProperStandard</code>
     * is <code>false</code>, then the compact (spaceless)
     * comma delimited DNs are transformed to lower case.
     * This method assumes that the DNs are case-insensitive.
     *
     * If it is not known whether the directory server
     * will return spaceless or spaced commas, we need
     * to use the full DN standardisation. Set
     * <code>forceProperStandard</code> to <code>true</code>
     * for the significantly slower, 100% effective,
     * standardisation. This method also assumes nothing
     * about the case of DNs.
     *
     * @param dn                  original DN.
     * @param forceProperStandard <code>true</code> if you
     *                            want to enforce the slow but effective standardisation process.
     * @return DN in standard form.
     */
    @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "returned as String")
    public static String standardise(String dn, boolean forceProperStandard) {
        if (forceProperStandard) {
            try {
                return standardise(new LdapName(dn), true);
            } catch (InvalidNameException e) {
            }
        }

        return toLowerCase(dn);
    }

    public static String standardise(LdapName name, boolean forceProperStandard) {
        if (forceProperStandard) {
            // Reconstruct the LdapName so LdapName#unparsed is null and toString will recreate it.
            return toLowerCase(new LdapName(name.getRdns()).toString());
        } else {
            return toLowerCase(name.toString());
        }
    }
}
