package com.atlassian.crowd.directory.ldap.util;

import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.Filter;
import org.springframework.ldap.filter.GreaterThanOrEqualsFilter;
import org.springframework.ldap.filter.NotPresentFilter;
import org.springframework.ldap.filter.OrFilter;

import java.util.Date;

/**
 * Utility for handling ActiveDirectory's account expiration setting.
 *
 * @see <a href="https://msdn.microsoft.com/en-us/library/ms675098%28v=vs.85%29.aspx">Account-Expires attribute</a>
 * @since v2.8.3
 */
public class ActiveDirectoryExpirationUtils {
    public static final String ACCOUNT_EXPIRES_ATTRIBUTE = "accountExpires";

    /**
     * Offset in 100-nanoseconds used to convert a Java date (number of milliseconds since January 1, 1970)
     * to ActiveDirectory's expiration date (number of 100-nanoseconds since January 1, 1601).
     * <p>
     * The offset was computed by converting six sample dates and matching the results with Microsoft's
     * <code>w32tm.exe /ntte timestamp</code>. (The reference utility in the Microsoft KB
     * about the date encoding: https://support.microsoft.com/en-us/kb/555936).
     *
     * @see #encodeDate(Date)
     */
    private static final long DATE_OFFSET = 116444736000000000L;

    public static Filter notExpiredAt(Date date) {
        OrFilter expirationFilter = new OrFilter();
        // AD uses accountExpires values of 0 and 0x7FFFFFFFFFFFFFFF to represent accounts that never expire.
        // This filter does not need to check the second value as it's the maximum positive long value and
        // will always be greater than or equal to the provided date.
        expirationFilter.or(new EqualsFilter(ACCOUNT_EXPIRES_ATTRIBUTE, "0"));
        expirationFilter.or(new NotPresentFilter(ACCOUNT_EXPIRES_ATTRIBUTE));
        expirationFilter.or(new GreaterThanOrEqualsFilter(ACCOUNT_EXPIRES_ATTRIBUTE, Long.toString(encodeDate(date))));
        return expirationFilter;
    }

    /**
     * Encode a date as "number of 100-nanosecond intervals since January 1, 1601 (UTC)".
     *
     * @see <a href="https://support.microsoft.com/en-us/kb/555936">How to convert date/time attributes in Active Directory</a>
     */
    public static long encodeDate(Date date) {
        // number of 100-nanosecond intervals since January 1, 1970 (GMT)
        long intervalsSinceJanuaryFirst1970 = date.getTime() * 10000;
        // number of 100-nanosecond intervals since January 1, 1601 (GMT)
        long intervalsSinceJanuaryFirst1601 = intervalsSinceJanuaryFirst1970 + DATE_OFFSET;
        // sanity check
        if (intervalsSinceJanuaryFirst1601 < 0) {
            throw new IllegalArgumentException("The date " + date + " could not be encoded");
        }
        return intervalsSinceJanuaryFirst1601;
    }
}
