package com.atlassian.crowd.directory.ldap.monitoring;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

public abstract class TimedSupplier<T> implements Supplier<T> {
    private static final Logger LOGGER = LoggerFactory.getLogger(TimedSupplier.class);
    private final String operationDescription;
    private final Stopwatch watch;
    private final long thresholdMillis;
    private final Logger log;

    public TimedSupplier(String operationDescription, long thresholdMillis) {
        this(operationDescription, Stopwatch.createUnstarted(), LOGGER, thresholdMillis);
    }

    @VisibleForTesting
    public TimedSupplier(String operationDescription, Stopwatch stopWatch, Logger log, long thresholdMillis) {
        this.operationDescription = operationDescription;
        this.watch = stopWatch;
        this.log = log;
        this.thresholdMillis = thresholdMillis;
    }

    public abstract T timedGet();

    @Override
    public final T get() {
        log.debug("Execute operation {}", operationDescription);
        watch.start();
        try {
            return timedGet();
        } finally {
            watch.stop();
            if (watch.elapsed(TimeUnit.MILLISECONDS) > thresholdMillis) {
                log.info("Timed call for {} took {}ms", operationDescription, watch.elapsed(TimeUnit.MILLISECONDS));
            } else if (log.isDebugEnabled()) {
                log.debug("Timed call for {} took {}ms", operationDescription, watch.elapsed(TimeUnit.MILLISECONDS));
            }
        }
    }
}
