package com.atlassian.crowd.directory.ldap.mapper;

import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.directory.ldap.mapper.attribute.AttributeMapper;
import com.atlassian.crowd.directory.ldap.mapper.entity.LDAPUserAttributesMapper;
import com.atlassian.crowd.directory.ldap.util.DNStandardiser;
import com.atlassian.crowd.model.user.LDAPUserWithAttributes;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.ldap.NamingException;
import org.springframework.ldap.core.DirContextAdapter;

import javax.naming.directory.Attributes;
import javax.naming.ldap.LdapName;

import java.util.List;
import java.util.Set;

/**
 * Translates information returned from an LDAP directory into a {@link com.atlassian.crowd.model.user.LDAPUserWithAttributes}
 * implementation of {@link com.atlassian.crowd.model.user.User}.
 */
public class UserContextMapper extends ContextMapperWithCustomAttributes<LDAPUserWithAttributes> {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    protected final long directoryId;
    protected final LDAPPropertiesMapper ldapPropertiesMapper;

    public UserContextMapper(long directoryId, LDAPPropertiesMapper ldapPropertiesMapper, List<AttributeMapper> customAttributeMappers) {
        super(customAttributeMappers);
        this.directoryId = directoryId;
        this.ldapPropertiesMapper = ldapPropertiesMapper;
    }

    @Override
    protected Set<String> getCoreRequiredLdapAttributes() {
        return getAttributesMapper().getRequiredLdapAttributes();
    }

    /**
     * Called by Spring LDAP on every object fetched from the LDAP directory.
     *
     * @param context A {@link org.springframework.ldap.core.DirContextAdapter DirContextAdapter} containing information about the object
     * @return {@link com.atlassian.crowd.model.user.LDAPUserWithAttributes}.
     */
    public LDAPUserWithAttributes mapFromContext(DirContextAdapter context) throws NamingException {
        Attributes attributes = context.getAttributes();
        LDAPUserAttributesMapper mapper = getAttributesMapper();

        // build user from common attributes
        MDC.put("crowd.ldap.context", context.getDn().toString());
        UserTemplateWithAttributes userTemplate;
        try {
            userTemplate = mapper.mapUserFromAttributes(attributes);
        } finally {
            MDC.remove("crowd.ldap.context");
        }

        // map custom attributes
        for (AttributeMapper attributeMapper : customAttributeMappers) {
            try {
                userTemplate.setAttribute(attributeMapper.getKey(), attributeMapper.getValues(context));
            } catch (Exception e) {
                logger.warn("Failed to map attribute <" + attributeMapper.getKey() + "> from context with DN <" + context.getDn().toString() + ">", e);
            }
        }

        String dn = DNStandardiser.standardise((LdapName) context.getDn(), !ldapPropertiesMapper.isRelaxedDnStandardisation());

        LDAPUserWithAttributes user = new LDAPUserWithAttributes(dn, userTemplate);

        if (logger.isTraceEnabled()) {
            logger.trace("Created user <" + user + "> from DN <" + context.getDn() + ">");
        }

        return user;
    }

    /**
     * Split out so it can be overriden.
     */
    protected LDAPUserAttributesMapper getAttributesMapper() {
        return new LDAPUserAttributesMapper(directoryId, ldapPropertiesMapper);
    }
}
