package com.atlassian.crowd.directory.ldap.mapper;

import java.util.Objects;

import javax.naming.Binding;
import javax.naming.NameClassPair;
import javax.naming.NamingException;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;

import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.ContextMapperCallbackHandler;
import org.springframework.ldap.core.DirContextAdapter;
import org.springframework.ldap.core.ObjectRetrievalException;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;


/**
 * <p>
 * Modified by Atlassian
 * From Spring LDAP 2.0.2.RELEASE
 * </p>
 * <p>
 * The purpose of this class is to provide a DirContextAdapter when no object was bound to the Binding returned.
 * This is to facilitate the fix for the <a href="https://jira.atlassian.com/browse/CWD-4754">CWD-4754</a> vulnerability,
 * which consists of setting the <code>returnObj</code> flag in SearchControls to false, so that JNDI's Java object
 * unmarshalling feature won't be used.
 * </p>
 * <p>
 * Changes:
 * - if no object was bound to the {@link javax.naming.Binding}, but the {@link javax.naming.NameClassPair} was an
 * instance of {@link javax.naming.directory.SearchResult}, then a {@link org.springframework.ldap.core.DirContextAdapter}
 * wrapping the {@link javax.naming.NameClassPair} will be returned.
 * </p>
 */
public class AttributeToContextCallbackHandler<T> extends
        ContextMapperCallbackHandler<T> {
    private final ContextMapper<T> mapper;

    public AttributeToContextCallbackHandler(final ContextMapper<T> mapper) {
        super(mapper);
        Objects.requireNonNull(mapper, "Mapper must not be empty");
        this.mapper = mapper;
    }

    @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "No user input")
    public T getObjectFromNameClassPair(NameClassPair nameClassPair) throws NamingException {
        if (!(nameClassPair instanceof Binding)) {
            throw new IllegalArgumentException("Parameter must be an instance of Binding");
        }

        Binding binding = (Binding) nameClassPair;
        Object object = binding.getObject();
        if (object == null) {
            if (nameClassPair instanceof SearchResult) {
                object = new DirContextAdapter(((SearchResult) nameClassPair).getAttributes(),
                        new LdapName(nameClassPair.getNameInNamespace()));
            } else {
                throw new ObjectRetrievalException(
                        "Binding did not contain any object.");
            }
        }
        return mapper.mapFromContext(object);
    }
}